<?php

use Elementor\Widget_Base;
use Elementor\Controls_Manager;

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}

class skill_comment_post_Widget extends Widget_Base {
    /**
     * Get widget name.
     */
    public function get_name() {
        return 'skill_comment_post';
    }

    /**
     * Get widget title.
     */
    public function get_title() {
        return __( 'کامنت پست', 'skill' );
    }

    /**
     * Get widget icon.
     */
    public function get_icon() {
        return 'eicon-comments';
    }

    /**
     * Get widget categories.
     */
    public function get_categories() {
        return [ 'skill-category' ];
    }

    /**
     * Register widget controls (settings in Elementor panel).
     */
    protected function _register_controls() {
        $this->start_controls_section(
            'section_settings',
            [
                'label' => __( 'تنظیمات', 'skill' ),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'post_id',
            [
                'label' => __( 'شناسه پست', 'skill' ),
                'type' => Controls_Manager::NUMBER,
                'default' => get_the_ID(),
                'description' => __( 'شناسه پستی که می‌خواهید کامنت‌های آن نمایش داده شود. خالی بگذارید برای پست فعلی.', 'skill' ),
            ]
        );

        $this->add_control(
            'show_comment_form',
            [
                'label' => __( 'نمایش فرم کامنت', 'skill' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => __( 'نمایش', 'skill' ),
                'label_off' => __( 'مخفی', 'skill' ),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'comments_per_page',
            [
                'label' => __( 'تعداد کامنت در هر صفحه', 'skill' ),
                'type' => Controls_Manager::NUMBER,
                'default' => 10,
                'min' => 1,
                'max' => 50,
            ]
        );

        $this->end_controls_section();

        // Style Section
        $this->start_controls_section(
            'section_style',
            [
                'label' => __( 'استایل', 'skill' ),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'container_background',
            [
                'label' => __( 'پس‌زمینه کانتینر', 'skill' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#0a090a',
                'selectors' => [
                    '{{WRAPPER}} .skill-comments-widget' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'container_border_radius',
            [
                'label' => __( 'گردی گوشه‌ها', 'skill' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px', '%' ],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 50,
                    ],
                    '%' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 14,
                ],
                'selectors' => [
                    '{{WRAPPER}} .skill-comments-widget' => 'border-radius: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'container_box_shadow',
                'selector' => '{{WRAPPER}} .skill-comments-widget',
            ]
        );

        $this->end_controls_section();
    }

    /**
     * Custom callback for displaying comments in a nested structure with custom HTML.
     */
    public function skill_comment_callback($comment, $args, $depth) {
        $GLOBALS['comment'] = $comment;
        $is_child = $depth > 1;
        ?>
        <li <?php comment_class('skill-comment-item' . ($is_child ? ' skill-comment-child' : '')); ?> id="comment-<?php comment_ID(); ?>">
            <div class="skill-comment-box">
                <div class="skill-comment-side">
                    <div class="skill-comment-author-avatar">
                        <?php echo get_avatar($comment, 48); ?>
                    </div>
                    <div class="skill-comment-author-name"><?php echo esc_html(get_comment_author()); ?></div>
                    <div class="skill-comment-date"><?php echo get_comment_date('Y/m/d'); ?></div>
                </div>
                <div class="skill-comment-main">
                    <div class="skill-comment-content"><?php comment_text(); ?></div>
                    <div class="skill-comment-actions">
                        <?php
                        comment_reply_link(array_merge($args, array(
                            'reply_text' => '<span class="skill-reply-icon">&#8634;</span> <span>' . __( 'پاسخ', 'skill' ) . '</span>',
                            'depth' => $depth,
                            'max_depth' => $args['max_depth'],
                            'add_below' => 'comment',
                            'before' => '',
                            'after' => '',
                        )));
                        ?>
                    </div>
                </div>
            </div>
        <?php
    }

    /**
     * Render widget output on the frontend.
     */
    protected function render() {
        $settings = $this->get_settings_for_display();
        $post_id = !empty($settings['post_id']) ? intval($settings['post_id']) : get_the_ID();
        ?>
        <div class="skill-comments-widget dark-theme" dir="rtl">
            <?php
            // Display comments list with custom callback
            $comments = get_comments([ 'post_id' => $post_id, 'status' => 'approve' ]);
            if ( $comments ) {
                echo '<ul class="skill-comments-list">';
                wp_list_comments([
                    'callback' => array($this, 'skill_comment_callback'),
                    'style' => 'ul',
                    'avatar_size' => 40,
                    'max_depth' => 3,
                    'reverse_top_level' => false,
                    'reverse_children' => false,
                ], $comments);
                echo '</ul>';
            } else {
                echo '<div class="skill-no-comments">' . __( 'هنوز دیدگاهی ثبت نشده است.', 'skill' ) . '</div>';
            }
            
            // Display comment form if enabled
            if ( $settings['show_comment_form'] === 'yes' ) {
                comment_form([
                    'title_reply' => __( 'ارسال دیدگاه', 'skill' ),
                    'label_submit' => __( 'ارسال دیدگاه', 'skill' ),
                    'comment_notes_before' => '',
                    'comment_notes_after' => '',
                    'class_submit' => 'skill-submit-btn',
                    'class_form' => 'skill-comment-form',
                    'logged_in_as' => '',
                    'id_form' => 'skill-commentform',
                    'id_submit' => 'skill-submit',
                    'title_reply_before' => '<h3 id="reply-title" class="comment-reply-title">',
                    'title_reply_after' => '</h3>',
                    'fields' => [
                        'author' => '<input id="author" name="author" type="text" placeholder="' . esc_attr__( 'نام شما*', 'skill' ) . '" required />',
                        'email'  => '<input id="email" name="email" type="email" placeholder="' . esc_attr__( 'ایمیل*', 'skill' ) . '" required />',
                        'url'    => '<input id="url" name="url" type="text" placeholder="' . esc_attr__( 'وبسایت (اختیاری)', 'skill' ) . '" />',
                    ],
                    'comment_field' => '<textarea id="comment" name="comment" placeholder="' . esc_attr__( 'متن دیدگاه*', 'skill' ) . '" required></textarea>',
                    'submit_button' => '<button name="submit" type="submit" id="skill-submit" class="skill-submit-btn">' . esc_html__( 'ارسال دیدگاه', 'skill' ) . '</button>',
                    'submit_field'  => '<div class="skill-submit-wrap">%1$s %2$s</div>',
                    'comment_notes_before' => '',
                    'comment_notes_after' => '',
                    'fields' => [
                        'author' => '<input id="author" name="author" type="text" placeholder="' . esc_attr__( 'نام شما*', 'skill' ) . '" required />',
                        'email'  => '<input id="email" name="email" type="email" placeholder="' . esc_attr__( 'ایمیل*', 'skill' ) . '" required />',
                        'url'    => '<input id="url" name="url" type="text" placeholder="' . esc_attr__( 'وبسایت (اختیاری)', 'skill' ) . '" />',
                        'cookies' => '',
                    ],
                ], $post_id );
            }
            ?>
        </div>
        <style>
            .skill-comments-widget.dark-theme {
                background: #0a090a;
                color: #fff;
                border-radius: 14px;
                padding: 32px 24px;
                margin: 24px auto;
                box-shadow: 0 2px 24px rgba(0,0,0,0.18);
                max-width: 1100px;
            }
            .skill-comments-list {
                list-style: none;
                padding: 0;
                margin: 0 0 32px 0;
            }
            .skill-comment-item {
                background: rgba(255, 255, 255, 0.03);
                backdrop-filter: blur(15px);
                -webkit-backdrop-filter: blur(15px);
                border: 1px solid rgba(255, 255, 255, 0.08);
                border-radius: 20px;
                padding: 24px;
                margin-bottom: 24px;
                position: relative;
                transition: all 0.3s ease;
                overflow: hidden;
            }
            .skill-comment-item::before {
                content: '';
                position: absolute;
                top: 0;
                left: 0;
                right: 0;
                bottom: 0;
                background: linear-gradient(135deg, rgba(14, 89, 242, 0.05), rgba(255, 255, 255, 0.02));
                border-radius: 20px;
                z-index: -1;
            }
            .skill-comment-item:hover {
                transform: translateY(-2px);
                box-shadow: 0 8px 25px rgba(14, 89, 242, 0.15);
                border-color: rgba(14, 89, 242, 0.2);
            }
            .skill-comment-child {
                background: rgba(255, 255, 255, 0.05);
                backdrop-filter: blur(20px);
                -webkit-backdrop-filter: blur(20px);
                border: 1px solid rgba(255, 255, 255, 0.1);
                border-radius: 18px;
                margin: 24px 40px 0 0;
                padding: 20px;
                position: relative;
                overflow: hidden;
            }
            .skill-comment-child::before {
                content: '';
                position: absolute;
                top: 0;
                left: 0;
                right: 0;
                bottom: 0;
                background: linear-gradient(135deg, rgba(14, 89, 242, 0.08), rgba(255, 255, 255, 0.03));
                border-radius: 18px;
                z-index: -1;
            }
            .skill-comment-child:hover {
                transform: translateY(-1px);
                box-shadow: 0 6px 20px rgba(14, 89, 242, 0.12);
            }
            .skill-comment-box {
                display: flex;
                flex-direction: row;
                align-items: flex-start;
                gap: 24px;
                position: relative;
            }
            .skill-comment-side {
                display: flex;
                flex-direction: column;
                align-items: center;
                min-width: 80px;
                margin-top: 4px;
            }
            .skill-comment-author-avatar {
                position: relative;
                margin-bottom: 12px;
            }
            .skill-comment-author-avatar img {
                border-radius: 50%;
                width: 56px;
                height: 56px;
                object-fit: cover;
                background: linear-gradient(135deg, #0e59f2, #00d4ff);
                border: 3px solid rgba(255, 255, 255, 0.1);
                box-shadow: 0 4px 15px rgba(14, 89, 242, 0.3);
                transition: all 0.3s ease;
            }
            .skill-comment-author-avatar::after {
                content: '';
                position: absolute;
                top: -2px;
                left: -2px;
                right: -2px;
                bottom: -2px;
                background: linear-gradient(135deg, #0e59f2, #00d4ff);
                border-radius: 50%;
                z-index: -1;
                opacity: 0.3;
                transition: opacity 0.3s ease;
            }
            .skill-comment-item:hover .skill-comment-author-avatar img {
                transform: scale(1.05);
                box-shadow: 0 6px 20px rgba(14, 89, 242, 0.4);
            }
            .skill-comment-item:hover .skill-comment-author-avatar::after {
                opacity: 0.5;
            }
            .skill-comment-author-name {
                color: #0e59f2;
                font-weight: 600;
                font-size: 1.1em;
                margin-bottom: 4px;
                text-align: center;
                text-shadow: 0 0 10px rgba(14, 89, 242, 0.3);
            }
            .skill-comment-date {
                color: rgba(255, 255, 255, 0.7);
                font-size: 0.9em;
                text-align: center;
                background: rgba(255, 255, 255, 0.05);
                padding: 4px 12px;
                border-radius: 12px;
                backdrop-filter: blur(10px);
                -webkit-backdrop-filter: blur(10px);
            }
            .skill-comment-main {
                flex: 1;
                display: flex;
                flex-direction: column;
                justify-content: space-between;
                min-width: 0;
            }
            .skill-comment-content {
                color: #fff;
                font-size: 1.05em;
                margin-bottom: 45px;
                word-break: break-word;
                line-height: 1.6;
                text-shadow: 0 0 5px rgba(255, 255, 255, 0.1);
            }
            .skill-comment-actions {
                display: flex;
                align-items: center;
                justify-content: flex-start;
                position: absolute;
                left: 24px;
                bottom: 16px;
            }
            .skill-reply-icon {
                color: #0e59f2;
                font-size: 1.1em;
                margin-left: 6px;
                transition: all 0.3s ease;
            }
            .skill-comment-actions a {
                color: #0e59f2;
                font-weight: 500;
                text-decoration: none;
                transition: all 0.3s ease;
                display: flex;
                align-items: center;
                gap: 6px;
                background: rgba(14, 89, 242, 0.1);
                padding: 8px 16px;
                border-radius: 20px;
                border: 1px solid rgba(14, 89, 242, 0.2);
                backdrop-filter: blur(10px);
                -webkit-backdrop-filter: blur(10px);
            }
            .skill-comment-actions a:hover {
                color: #fff;
                background: rgba(14, 89, 242, 0.3);
                transform: translateY(-1px);
                box-shadow: 0 4px 12px rgba(14, 89, 242, 0.3);
            }
            .skill-comment-actions a:hover .skill-reply-icon {
                transform: rotate(180deg);
            }
            .skill-no-comments {
                color: rgba(255, 255, 255, 0.8);
                margin-bottom: 32px;
                text-align: center;
                padding: 40px 20px;
                background: rgba(255, 255, 255, 0.03);
                backdrop-filter: blur(15px);
                -webkit-backdrop-filter: blur(15px);
                border: 1px solid rgba(255, 255, 255, 0.08);
                border-radius: 20px;
                position: relative;
                overflow: hidden;
            }
            .skill-no-comments::before {
                content: '';
                position: absolute;
                top: 0;
                left: 0;
                right: 0;
                bottom: 0;
                background: linear-gradient(135deg, rgba(14, 89, 242, 0.05), rgba(255, 255, 255, 0.02));
                border-radius: 20px;
                z-index: -1;
            }
            .skill-no-comments::after {
                content: '💬';
                font-size: 3em;
                display: block;
                margin-bottom: 16px;
                opacity: 0.6;
            }
            .skill-comment-form {
                margin-top: 32px;
                background: rgba(255, 255, 255, 0.05);
                backdrop-filter: blur(20px);
                -webkit-backdrop-filter: blur(20px);
                border: 1px solid rgba(255, 255, 255, 0.1);
                border-radius: 20px;
                padding: 32px;
                position: relative;
                overflow: hidden;
            }
            .skill-comment-form::before {
                content: '';
                position: absolute;
                top: 0;
                left: 0;
                right: 0;
                bottom: 0;
                background: linear-gradient(135deg, rgba(14, 89, 242, 0.1), rgba(255, 255, 255, 0.05));
                border-radius: 20px;
                z-index: -1;
            }
            .skill-comment-form h3 {
                color: #fff;
                margin-bottom: 24px;
                font-size: 1.5em;
                font-weight: 600;
                text-align: center;
                position: relative;
            }
            .skill-comment-form h3::after {
                content: '';
                position: absolute;
                bottom: -8px;
                left: 50%;
                transform: translateX(-50%);
                width: 60px;
                height: 3px;
                background: linear-gradient(90deg, #0e59f2, #00d4ff);
                border-radius: 2px;
            }
            .skill-comment-form input,
            .skill-comment-form textarea {
                background: rgba(255, 255, 255, 0.08);
                color: #ffffff;
                border: 2px solid rgba(255, 255, 255, 0.1);
                border-radius: 15px;
                padding: 16px 20px;
                width: 100%;
                margin-bottom: 20px;
                font-size: 14px;
                transition: all 0.3s ease;
                backdrop-filter: blur(10px);
                -webkit-backdrop-filter: blur(10px);
            }
            .skill-comment-form input:focus,
            .skill-comment-form textarea:focus {
                outline: none;
                border-color: rgba(14, 89, 242, 0.6);
                box-shadow: 0 0 20px rgba(14, 89, 242, 0.3);
                background: rgba(255, 255, 255, 0.12);
                transform: translateY(-2px);
            }
            .skill-comment-form input::placeholder,
            .skill-comment-form textarea::placeholder {
                color: rgba(255, 255, 255, 0.6);
                font-weight: 300;
            }
            .skill-comment-form textarea {
                min-height: 140px;
                resize: vertical;
                line-height: 1.6;
            }
            /* Email field: placeholder RTL, typed LTR */
            .skill-comment-form input[type="email"] {
                text-align: left;
                direction: ltr;
            }
            .skill-comment-form input[type="email"]:placeholder-shown {
                text-align: right;
                direction: rtl;
            }
            .skill-comment-form input[type="email"]:focus {
                text-align: left;
                direction: ltr;
            }
            .skill-comment-form input[type="email"]:not(:placeholder-shown) {
                text-align: left;
                direction: ltr;
            }
            .skill-submit-btn {
                background: linear-gradient(135deg, #0e59f2, #00d4ff);
                color: #fff;
                border: none;
                padding: 15px 50px 15px 50px;
                border-radius: 50px;
                font-weight: 600;
                cursor: pointer;
                transition: all 0.4s ease;
                font-size: 16px;
                position: relative;
                overflow: hidden;
                box-shadow: 0 8px 25px rgba(14, 89, 242, 0.3);
                text-transform: uppercase;
                letter-spacing: 1px;
            }
            .skill-submit-btn::before {
                content: '';
                position: absolute;
                top: 0;
                left: -100%;
                width: 100%;
                height: 100%;
                background: linear-gradient(90deg, transparent, rgba(255, 255, 255, 0.2), transparent);
                transition: left 0.5s ease;
            }
            .skill-submit-btn:hover {
                background: linear-gradient(135deg, #00d4ff, #0e59f2);
                transform: translateY(-3px);
                box-shadow: 0 12px 35px rgba(14, 89, 242, 0.4);
            }
            .skill-submit-btn:hover::before {
                left: 100%;
            }
            .skill-submit-btn:active {
                transform: translateY(-1px);
                box-shadow: 0 6px 20px rgba(14, 89, 242, 0.3);
            }
            
            .skill-comment-form input[type="checkbox"],
            .skill-comment-form input[type="radio"] {
                display: none !important;
            }
            
            .skill-comment-form .comment-form-cookies-consent {
                display: none !important;
            }
            
            .skill-comment-form .comment-notes {
                display: none !important;
            }
            
            .skill-comment-form .logged-in-as {
                display: none !important;
            }
            
            .skill-comment-form .comment-form-author,
            .skill-comment-form .comment-form-email,
            .skill-comment-form .comment-form-url,
            .skill-comment-form .comment-form-comment {
                margin-bottom: 20px;
            }
            
            .skill-comment-form .comment-form-author input,
            .skill-comment-form .comment-form-email input,
            .skill-comment-form .comment-form-url input {
                width: 100%;
                box-sizing: border-box;
            }
            
            @media (max-width: 768px) {
                .skill-comment-form {
                    padding: 20px;
                }
                
                .skill-submit-btn {
                    width: 100%;
                    padding: 14px 20px;
                }
            }
            @media (max-width: 700px) {
                .skill-comments-widget.dark-theme {
                    padding: 10px 2px;
                }
                .skill-comment-child {
                    margin: 16px 0 0 0;
                    padding: 16px 8px 0 0;
                }
                .skill-comment-box {
                    flex-direction: column;
                    gap: 8px;
                }
                .skill-comment-side {
                    flex-direction: row-reverse;
                    align-items: center;
                    gap: 8px;
                    min-width: 0;
                }
                .skill-comment-main {
                    margin-top: 8px;
                }
                .skill-comment-actions {
                    left: 8px;
                    bottom: 8px;
                }
            }
        </style>
        <?php
    }
}