<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Prevent direct access

class skill_counter_animation_Widget extends \Elementor\Widget_Base {

    public function get_name() {
        return 'skill_counter_animation';
    }

    public function get_title() {
        return esc_html__( 'شمارنده انیمیشن', 'skill' );
    }

    public function get_icon() {
        return 'eicon-number-field';
    }

    public function get_categories() {
        return [ 'skill-category' ];
    }

    public function get_keywords() {
        return [ 'شمارنده', 'انیمیشن', 'counter', 'animation', 'آمار' ];
    }

    protected function register_controls() {
        // Content Section
        $this->start_controls_section(
            'content_section',
            [
                'label' => esc_html__( 'محتوای شمارنده', 'skill' ),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        // Header controls (mirroring Skill_title_desc_Widget)
        $this->add_control(
            'show_title',
            [
                'label' => esc_html__( 'نمایش عنوان', 'skill' ),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'بله', 'skill' ),
                'label_off' => esc_html__( 'خیر', 'skill' ),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'title_text',
            [
                'label' => esc_html__( 'عنوان', 'skill' ),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => esc_html__( 'آمار و دستاوردها', 'skill' ),
                'placeholder' => esc_html__( 'عنوان را وارد کنید', 'skill' ),
                'condition' => [ 'show_title' => 'yes' ],
            ]
        );

        $this->add_control(
            'show_dot',
            [
                'label' => esc_html__( 'نمایش مربع آبی', 'skill' ),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'بله', 'skill' ),
                'label_off' => esc_html__( 'خیر', 'skill' ),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'dot_position',
            [
                'label' => esc_html__( 'جایگاه مربع', 'skill' ),
                'type' => \Elementor\Controls_Manager::SELECT,
                'options' => [
                    'right' => esc_html__( 'راست عنوان', 'skill' ),
                    'left'  => esc_html__( 'چپ عنوان', 'skill' ),
                ],
                'default' => 'right',
                'condition' => [ 'show_dot' => 'yes' ],
            ]
        );

        $this->add_control(
            'show_desc',
            [
                'label' => esc_html__( 'نمایش توضیح کوتاه', 'skill' ),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'بله', 'skill' ),
                'label_off' => esc_html__( 'خیر', 'skill' ),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'desc_text',
            [
                'label' => esc_html__( 'توضیح کوتاه', 'skill' ),
                'type' => \Elementor\Controls_Manager::TEXTAREA,
                'default' => esc_html__( 'توضیحات مربوط به آمار و دستاوردها', 'skill' ),
                'rows' => 2,
                'condition' => [ 'show_desc' => 'yes' ],
            ]
        );

        // Repeater for counter items
        $repeater = new \Elementor\Repeater();

        $repeater->add_control(
            'counter_number',
            [
                'label' => esc_html__( 'عدد شمارنده', 'skill' ),
                'type' => \Elementor\Controls_Manager::NUMBER,
                'default' => 80,
                'min' => 0,
                'step' => 1,
            ]
        );

        $repeater->add_control(
            'counter_text',
            [
                'label' => esc_html__( 'متن شمارنده', 'skill' ),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => esc_html__( 'پروژه موفق', 'skill' ),
                'placeholder' => esc_html__( 'مثال: پروژه موفق', 'skill' ),
            ]
        );

        $repeater->add_control(
            'icon',
            [
                'label' => esc_html__( 'آیکون', 'skill' ),
                'type' => \Elementor\Controls_Manager::ICONS,
                'default' => [
                    'value' => 'fas fa-check-circle',
                    'library' => 'fa-solid',
                ],
            ]
        );

        $repeater->add_control(
            'animation_delay',
            [
                'label' => esc_html__( 'تاخیر انیمیشن (ثانیه)', 'skill' ),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => [ 's' ],
                'range' => [
                    's' => [
                        'min' => 0,
                        'max' => 5,
                        'step' => 0.1,
                    ],
                ],
                'default' => [
                    'unit' => 's',
                    'size' => 0.2,
                ],
            ]
        );

        $this->add_control(
            'counter_items',
            [
                'label' => esc_html__( 'آیتم‌های شمارنده', 'skill' ),
                'type' => \Elementor\Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'counter_number' => 80,
                        'counter_text' => esc_html__( 'پروژه موفق', 'skill' ),
                        'icon' => [
                            'value' => 'fas fa-check-circle',
                            'library' => 'fa-solid',
                        ],
                        'animation_delay' => [ 'size' => 0.2, 'unit' => 's' ],
                    ],
                    [
                        'counter_number' => 150,
                        'counter_text' => esc_html__( 'مشتری راضی', 'skill' ),
                        'icon' => [
                            'value' => 'fas fa-smile',
                            'library' => 'fa-solid',
                        ],
                        'animation_delay' => [ 'size' => 0.4, 'unit' => 's' ],
                    ],
                    [
                        'counter_number' => 5,
                        'counter_text' => esc_html__( 'سال تجربه', 'skill' ),
                        'icon' => [
                            'value' => 'fas fa-clock',
                            'library' => 'fa-solid',
                        ],
                        'animation_delay' => [ 'size' => 0.6, 'unit' => 's' ],
                    ],
                    [
                        'counter_number' => 24,
                        'counter_text' => esc_html__( 'ساعت پشتیبانی', 'skill' ),
                        'icon' => [
                            'value' => 'fas fa-headset',
                            'library' => 'fa-solid',
                        ],
                        'animation_delay' => [ 'size' => 0.8, 'unit' => 's' ],
                    ],
                ],
                'title_field' => '{{{ counter_text }}}',
            ]
        );

        $this->end_controls_section();

        // Animation Settings
        $this->start_controls_section(
            'animation_section',
            [
                'label' => esc_html__( 'تنظیمات انیمیشن', 'skill' ),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'animation_type',
            [
                'label' => esc_html__( 'نوع انیمیشن', 'skill' ),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'fadeInUp',
                'options' => [
                    'fadeInUp' => esc_html__( 'ظاهر شدن از پایین', 'skill' ),
                    'fadeInDown' => esc_html__( 'ظاهر شدن از بالا', 'skill' ),
                    'fadeInLeft' => esc_html__( 'ظاهر شدن از چپ', 'skill' ),
                    'fadeInRight' => esc_html__( 'ظاهر شدن از راست', 'skill' ),
                    'zoomIn' => esc_html__( 'بزرگنمایی', 'skill' ),
                    'slideInUp' => esc_html__( 'اسلاید از پایین', 'skill' ),
                    'bounceIn' => esc_html__( 'پرش', 'skill' ),
                    'flipInX' => esc_html__( 'چرخش افقی', 'skill' ),
                ],
            ]
        );

        $this->add_control(
            'animation_duration',
            [
                'label' => esc_html__( 'مدت انیمیشن (ثانیه)', 'skill' ),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => [ 's' ],
                'range' => [
                    's' => [
                        'min' => 0.1,
                        'max' => 3,
                        'step' => 0.1,
                    ],
                ],
                'default' => [
                    'unit' => 's',
                    'size' => 1,
                ],
            ]
        );

        $this->add_control(
            'counter_duration',
            [
                'label' => esc_html__( 'مدت شمارش (ثانیه)', 'skill' ),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => [ 's' ],
                'range' => [
                    's' => [
                        'min' => 0.5,
                        'max' => 5,
                        'step' => 0.1,
                    ],
                ],
                'default' => [
                    'unit' => 's',
                    'size' => 2,
                ],
            ]
        );

        $this->add_control(
            'enable_counter_animation',
            [
                'label' => esc_html__( 'فعال‌سازی انیمیشن شمارنده', 'skill' ),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'فعال', 'skill' ),
                'label_off' => esc_html__( 'غیرفعال', 'skill' ),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->end_controls_section();

        // Layout Settings
        $this->start_controls_section(
            'layout_section',
            [
                'label' => esc_html__( 'تنظیمات چیدمان', 'skill' ),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_responsive_control(
            'columns',
            [
                'label' => esc_html__( 'تعداد ستون‌ها', 'skill' ),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => '4',
                'tablet_default' => '2',
                'mobile_default' => '1',
                'options' => [
                    '1' => esc_html__( '1 ستون', 'skill' ),
                    '2' => esc_html__( '2 ستون', 'skill' ),
                    '3' => esc_html__( '3 ستون', 'skill' ),
                    '4' => esc_html__( '4 ستون', 'skill' ),
                    '6' => esc_html__( '6 ستون', 'skill' ),
                ],
                'selectors' => [
                    '{{WRAPPER}} .skill-counter-item' => 'flex: 0 0 calc(100% / {{VALUE}});',
                ],
            ]
        );

        $this->add_control(
            'alignment',
            [
                'label' => esc_html__( 'تراز متن', 'skill' ),
                'type' => \Elementor\Controls_Manager::CHOOSE,
                'options' => [
                    'left' => [
                        'title' => esc_html__( 'چپ', 'skill' ),
                        'icon' => 'eicon-text-align-left',
                    ],
                    'center' => [
                        'title' => esc_html__( 'وسط', 'skill' ),
                        'icon' => 'eicon-text-align-center',
                    ],
                    'right' => [
                        'title' => esc_html__( 'راست', 'skill' ),
                        'icon' => 'eicon-text-align-right',
                    ],
                ],
                'default' => 'center',
                'selectors' => [
                    '{{WRAPPER}} .skill-counter-item' => 'text-align: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Style Section - Header (match title+desc widget)
        $this->start_controls_section(
            'tdw_style',
            [
                'label' => esc_html__( 'استایل عنوان و توضیح', 'skill' ),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control('tdw_align',[
            'label' => esc_html__('چینش','skill'),
            'type' => \Elementor\Controls_Manager::CHOOSE,
            'options' => [
                'right' => ['title'=>esc_html__('راست','skill'),'icon'=>'eicon-text-align-right'],
                'center'=> ['title'=>esc_html__('وسط','skill'),'icon'=>'eicon-text-align-center'],
                'left'  => ['title'=>esc_html__('چپ','skill'),'icon'=>'eicon-text-align-left'],
            ],
            'default' => 'center',
            'selectors' => [ '{{WRAPPER}} .tdw-wrap' => 'text-align: {{VALUE}};' ]
        ]);

        $this->add_control('title_color',[
            'label'=>esc_html__('رنگ عنوان','skill'),
            'type'=>\Elementor\Controls_Manager::COLOR,
            'default'=>'#ffffff',
            'selectors'=>[ '{{WRAPPER}} .tdw-title' => 'color: {{VALUE}};' ]
        ]);

        $this->add_group_control(\Elementor\Group_Control_Typography::get_type(),[
            'name'=>'title_typo',
            'label'=>esc_html__('تایپوگرافی عنوان','skill'),
            'selector'=>'{{WRAPPER}} .tdw-title'
        ]);

        $this->add_control('desc_color',[
            'label'=>esc_html__('رنگ توضیح','skill'),
            'type'=>\Elementor\Controls_Manager::COLOR,
            'default'=>'rgba(255,255,255,0.8)',
            'selectors'=>[ '{{WRAPPER}} .tdw-desc' => 'color: {{VALUE}};' ]
        ]);

        $this->add_group_control(\Elementor\Group_Control_Typography::get_type(),[
            'name'=>'desc_typo',
            'label'=>esc_html__('تایپوگرافی توضیح','skill'),
            'selector'=>'{{WRAPPER}} .tdw-desc'
        ]);

        $this->add_control('dot_color',[
            'label'=>esc_html__('رنگ مربع','skill'),
            'type'=>\Elementor\Controls_Manager::COLOR,
            'default'=>'#1e73be',
            'selectors'=>[ '{{WRAPPER}} .tdw-dot' => 'background-color: {{VALUE}};' ]
        ]);

        $this->add_control('dot_size',[
            'label'=>esc_html__('سایز مربع (px)','skill'),
            'type'=>\Elementor\Controls_Manager::SLIDER,
            'size_units'=>['px'],
            'range'=>['px'=>['min'=>6,'max'=>24]],
            'default'=>['unit'=>'px','size'=>10],
            'selectors'=>[ '{{WRAPPER}} .tdw-dot' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};' ]
        ]);

        $this->end_controls_section();

        // Style Section - Counter Items
        $this->start_controls_section(
            'style_counter_section',
            [
                'label' => esc_html__( 'آیتم‌های شمارنده', 'skill' ),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'counter_number_typography',
                'label' => esc_html__( 'تایپوگرافی عدد', 'skill' ),
                'selector' => '{{WRAPPER}} .skill-counter-number',
            ]
        );

        $this->add_control(
            'counter_number_color',
            [
                'label' => esc_html__( 'رنگ عدد', 'skill' ),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .skill-counter-number' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'use_gradient_number',
            [
                'label' => esc_html__( 'استفاده از گرادیانت برای عدد', 'skill' ),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'بله', 'skill' ),
                'label_off' => esc_html__( 'خیر', 'skill' ),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'counter_text_typography',
                'label' => esc_html__( 'تایپوگرافی متن', 'skill' ),
                'selector' => '{{WRAPPER}} .skill-counter-text',
            ]
        );

        $this->add_control(
            'counter_text_color',
            [
                'label' => esc_html__( 'رنگ متن', 'skill' ),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .skill-counter-text' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'icon_color',
            [
                'label' => esc_html__( 'رنگ آیکون', 'skill' ),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#ffffff',
                'selectors' => [
                    '{{WRAPPER}} .skill-counter-icon i' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .skill-counter-icon svg' => 'fill: {{VALUE}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'icon_size',
            [
                'label' => esc_html__( 'اندازه آیکون', 'skill' ),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => [ 'px', 'em' ],
                'range' => [
                    'px' => [
                        'min' => 10,
                        'max' => 100,
                        'step' => 1,
                    ],
                    'em' => [
                        'min' => 0.5,
                        'max' => 5,
                        'step' => 0.1,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 40,
                ],
                'selectors' => [
                    '{{WRAPPER}} .skill-counter-icon i' => 'font-size: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .skill-counter-icon svg' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'icon_container_size',
            [
                'label' => esc_html__( 'اندازه کانتینر آیکون', 'skill' ),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => [ 'px', 'em' ],
                'range' => [
                    'px' => [
                        'min' => 40,
                        'max' => 150,
                        'step' => 5,
                    ],
                    'em' => [
                        'min' => 2,
                        'max' => 10,
                        'step' => 0.2,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 60,
                ],
                'selectors' => [
                    '{{WRAPPER}} .skill-counter-icon' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'counter_item_padding',
            [
                'label' => esc_html__( 'فاصله داخلی', 'skill' ),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .skill-counter-item' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'counter_item_margin',
            [
                'label' => esc_html__( 'فاصله خارجی', 'skill' ),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .skill-counter-item' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Style Section - Container
        $this->start_controls_section(
            'style_container_section',
            [
                'label' => esc_html__( 'کانتینر', 'skill' ),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'container_bg_color',
            [
                'label' => esc_html__( 'رنگ پس‌زمینه اول', 'skill' ),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#667eea',
                'selectors' => [
                    '{{WRAPPER}} .skill-counter-container' => 'background: linear-gradient({{container_gradient_angle.SIZE}}deg, {{VALUE}} 0%, {{container_bg_color_second.VALUE}} 100%) !important;',
                ],
            ]
        );

        $this->add_control(
            'container_bg_color_second',
            [
                'label' => esc_html__( 'رنگ پس‌زمینه دوم', 'skill' ),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#764ba2',
                'selectors' => [
                    '{{WRAPPER}} .skill-counter-container' => 'background: linear-gradient({{container_gradient_angle.SIZE}}deg, {{container_bg_color.VALUE}} 0%, {{VALUE}} 100%) !important;',
                ],
            ]
        );

        $this->add_control(
            'container_gradient_angle',
            [
                'label' => esc_html__( 'زاویه گرادیانت', 'skill' ),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => [ 'deg' ],
                'range' => [
                    'deg' => [
                        'min' => 0,
                        'max' => 360,
                        'step' => 1,
                    ],
                ],
                'default' => [
                    'unit' => 'deg',
                    'size' => 135,
                ],
                'selectors' => [
                    '{{WRAPPER}} .skill-counter-container' => 'background: linear-gradient({{SIZE}}{{UNIT}}, {{container_bg_color.VALUE}} 0%, {{container_bg_color_second.VALUE}} 100%) !important;',
                ],
            ]
        );

        $this->add_control(
            'container_opacity',
            [
                'label' => esc_html__( 'شفافیت', 'skill' ),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => [ '%' ],
                'range' => [
                    '%' => [
                        'min' => 0,
                        'max' => 100,
                        'step' => 1,
                    ],
                ],
                'default' => [
                    'unit' => '%',
                    'size' => 100,
                ],
                'selectors' => [
                    '{{WRAPPER}} .skill-counter-container' => 'opacity: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name' => 'container_background',
                'label' => esc_html__( 'پس‌زمینه پیشرفته', 'skill' ),
                'types' => [ 'classic', 'gradient' ],
                'selector' => '{{WRAPPER}} .skill-counter-container',
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name' => 'container_border',
                'label' => esc_html__( 'حاشیه', 'skill' ),
                'selector' => '{{WRAPPER}} .skill-counter-container',
            ]
        );

        $this->add_responsive_control(
            'container_padding',
            [
                'label' => esc_html__( 'فاصله داخلی', 'skill' ),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .skill-counter-container' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'container_margin',
            [
                'label' => esc_html__( 'فاصله خارجی', 'skill' ),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .skill-counter-container' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        
        // Get animation settings with null checks
        $animation_type = isset($settings['animation_type']) ? $settings['animation_type'] : 'fadeInUp';
        $animation_duration = isset($settings['animation_duration']['size']) ? $settings['animation_duration']['size'] : 1;
        $counter_duration = isset($settings['counter_duration']['size']) ? $settings['counter_duration']['size'] : 2;
        $enable_counter_animation = isset($settings['enable_counter_animation']) ? $settings['enable_counter_animation'] : 'yes';
        $use_gradient_number = isset($settings['use_gradient_number']) ? $settings['use_gradient_number'] : 'yes';
        
        // Generate unique ID for this widget instance
        $widget_id = 'skill-counter-' . $this->get_id();
        ?>
        
        <div class="skill-counter-container <?php echo ($use_gradient_number === 'yes' ? 'use-gradient-number' : ''); ?>" id="<?php echo esc_attr($widget_id); ?>">
            <div class="tdw-wrap" dir="rtl">
                <div class="tdw-title-row" style="display:inline-flex; align-items:center; gap:8px; <?php echo (($settings['dot_position'] ?? 'right') === 'right') ? 'flex-direction: row-reverse;' : ''; ?>">
                    <?php if (!empty($settings['show_title']) && $settings['show_title'] === 'yes' && !empty($settings['title_text'])) : ?>
                        <h2 class="tdw-title"><?php echo esc_html($settings['title_text']); ?></h2>
                    <?php endif; ?>
                    <?php if (!empty($settings['show_dot']) && $settings['show_dot'] === 'yes') : ?>
                        <span class="tdw-dot" aria-hidden="true" style="display:inline-block; border-radius:3px;"></span>
                    <?php endif; ?>
                </div>
                <?php if (!empty($settings['show_desc']) && $settings['show_desc'] === 'yes' && !empty($settings['desc_text'])) : ?>
                    <div class="tdw-desc" style="margin-top:10px;">
                        <?php echo esc_html($settings['desc_text']); ?>
                    </div>
                <?php endif; ?>
            </div>
            
            <div class="skill-counter-items">
                <?php if (isset($settings['counter_items']) && is_array($settings['counter_items'])) : ?>
                    <?php foreach ($settings['counter_items'] as $index => $item) : ?>
                    <div class="skill-counter-item" 
                         data-number="<?php echo esc_attr(isset($item['counter_number']) ? $item['counter_number'] : 0); ?>"
                         data-delay="<?php echo esc_attr(isset($item['animation_delay']['size']) ? $item['animation_delay']['size'] : 0.2); ?>"
                         data-duration="<?php echo esc_attr($counter_duration); ?>"
                         data-animation="<?php echo esc_attr($animation_type); ?>"
                         data-animation-duration="<?php echo esc_attr($animation_duration); ?>">
                        
                        <?php if (isset($item['icon']['value']) && !empty($item['icon']['value'])) : ?>
                            <div class="skill-counter-icon">
                                <?php \Elementor\Icons_Manager::render_icon($item['icon'], ['aria-hidden' => 'true', 'class' => 'skill-counter-icon-element']); ?>
                            </div>
                        <?php endif; ?>
                        
                        <div class="skill-counter-content">
                            <div class="skill-counter-number" data-target="<?php echo esc_attr(isset($item['counter_number']) ? $item['counter_number'] : 0); ?>">
                                <?php echo esc_html(isset($item['counter_number']) ? $item['counter_number'] : 0); ?>
                            </div>
                            <div class="skill-counter-text">
                                <?php echo esc_html(isset($item['counter_text']) ? $item['counter_text'] : ''); ?>
                            </div>
                        </div>
                    </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>

        <style>
            .skill-counter-container {
                width: 100%;
                max-width: 1200px;
                margin: 0 auto;
                padding: 30px 20px;
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                border-radius: 20px;
                position: relative;
                overflow: hidden;
            }
            
            .skill-counter-container::before {
                content: '';
                position: absolute;
                top: 0;
                left: 0;
                right: 0;
                bottom: 0;
                background: linear-gradient(45deg, transparent 30%, rgba(255,255,255,0.1) 50%, transparent 70%);
                animation: shimmer 3s infinite;
                pointer-events: none;
            }
            
            @keyframes shimmer {
                0% { transform: translateX(-100%); }
                100% { transform: translateX(100%); }
            }
            
            .skill-counter-header {
                text-align: center;
                margin-bottom: 20px;
                position: relative;
                z-index: 2;
            }
            
            .skill-counter-main-title {
                font-size: 2.5rem;
                font-weight: bold;
                margin-bottom: 15px;
                color: #fff;
                text-shadow: 0 2px 4px rgba(0,0,0,0.3);
            }
            
            .skill-counter-description {
                font-size: 1.1rem;
                color: rgba(255,255,255,0.9);
                line-height: 1.6;
            }
            
            .skill-counter-items {
                display: flex;
                flex-wrap: wrap;
                gap: 24px;
                justify-content: center;
                position: relative;
                z-index: 2;
            }
            
                        .skill-counter-item {
                flex: 1;
                min-width: 220px;
                text-align: center;
                padding: 25px 20px;
                background: linear-gradient(145deg, rgba(255, 255, 255, 0.12) 0%, rgba(255, 255, 255, 0.06) 50%, rgba(255, 255, 255, 0.02) 100%);
                backdrop-filter: blur(25px);
                -webkit-backdrop-filter: blur(25px);
                border: 1px solid rgba(255, 255, 255, 0.15);
                border-radius: 20px;
                transition: all 0.5s cubic-bezier(0.4, 0, 0.2, 1);
                opacity: 0;
                transform: translateY(30px);
                position: relative;
                overflow: hidden;
            }
            
            .skill-counter-item::before {
                content: '';
                position: absolute;
                top: 0;
                left: -100%;
                width: 100%;
                height: 100%;
                background: linear-gradient(90deg, transparent, rgba(255,255,255,0.4), transparent);
                transition: left 0.5s ease;
            }
            
            .skill-counter-item:hover::before {
                left: 100%;
            }
            
            .skill-counter-item:hover {
                transform: translateY(-12px) scale(1.05);
                box-shadow: 
                    0 30px 60px rgba(0, 0, 0, 0.3),
                    0 8px 20px rgba(0, 0, 0, 0.2),
                    inset 0 1px 0 rgba(255, 255, 255, 0.3);
                background: linear-gradient(145deg, rgba(255, 255, 255, 0.2) 0%, rgba(255, 255, 255, 0.1) 50%, rgba(255, 255, 255, 0.05) 100%);
                border-color: rgba(255, 255, 255, 0.4);
                backdrop-filter: blur(30px);
                -webkit-backdrop-filter: blur(30px);
            }
            
            .skill-counter-item::after {
                content: '';
                position: absolute;
                top: 0;
                left: 0;
                right: 0;
                bottom: 0;
                background: linear-gradient(45deg, transparent 30%, rgba(255,255,255,0.1) 50%, transparent 70%);
                opacity: 0;
                transition: opacity 0.3s ease;
            }
            
            .skill-counter-item:hover::after {
                opacity: 1;
                animation: lightning 0.6s ease;
            }
            
            @keyframes lightning {
                0% { opacity: 0; }
                50% { opacity: 1; }
                100% { opacity: 0; }
            }
            
            .skill-counter-icon {
                margin-bottom: 20px;
                position: relative;
                z-index: 2;
                display: inline-flex;
                align-items: center;
                justify-content: center;
                padding: 12px;
                background: linear-gradient(135deg, rgba(255, 255, 255, 0.15) 0%, rgba(255, 255, 255, 0.05) 100%);
                border-radius: 50%;
                border: 1px solid rgba(255, 255, 255, 0.2);
                box-shadow: 
                    0 4px 12px rgba(0, 0, 0, 0.1),
                    inset 0 1px 0 rgba(255, 255, 255, 0.1);
                transition: all 0.4s ease;
                width: 60px;
                height: 60px;
                min-width: 60px;
                min-height: 60px;
            }
            
            .skill-counter-icon i,
            .skill-counter-icon svg {
                text-shadow: 0 2px 4px rgba(0,0,0,0.3);
                transition: all 0.3s ease;
                display: block;
                width: 100%;
                height: 100%;
                color: #ffffff;
            }
            
            .skill-counter-icon svg {
                fill: #ffffff;
            }
            
            .skill-counter-icon-element {
                display: block !important;
                width: 100% !important;
                height: 100% !important;
            }
            
            .skill-counter-item:hover .skill-counter-icon {
                transform: scale(1.1) rotate(5deg);
                background: linear-gradient(135deg, rgba(255, 255, 255, 0.25) 0%, rgba(255, 255, 255, 0.1) 100%);
                border-color: rgba(255, 255, 255, 0.4);
                box-shadow: 
                    0 8px 20px rgba(0, 0, 0, 0.2),
                    inset 0 1px 0 rgba(255, 255, 255, 0.2);
            }
            
            .skill-counter-item:hover .skill-counter-icon i,
            .skill-counter-item:hover .skill-counter-icon svg {
                transform: scale(1.15);
                text-shadow: 0 4px 8px rgba(0,0,0,0.4);
            }
            
            .skill-counter-number {
                font-size: 2.8rem;
                font-weight: 800;
                color: #fff;
                margin-bottom: 12px;
                line-height: 1;
                text-shadow: 0 2px 4px rgba(0,0,0,0.3);
                transition: all 0.4s ease;
            }

            /* Apply gradient to numbers only when enabled */
            .skill-counter-container.use-gradient-number .skill-counter-number {
                background: linear-gradient(135deg, #ffffff 0%, #f0f0f0 100%);
                -webkit-background-clip: text;
                -webkit-text-fill-color: transparent;
                background-clip: text;
            }
            
            .skill-counter-item:hover .skill-counter-number {
                transform: scale(1.08);
                text-shadow: 0 4px 8px rgba(0,0,0,0.4);
            }

            .skill-counter-container.use-gradient-number .skill-counter-item:hover .skill-counter-number {
                background: linear-gradient(135deg, #ffffff 0%, #e8f4fd 50%, #ffffff 100%);
                -webkit-background-clip: text;
                -webkit-text-fill-color: transparent;
                background-clip: text;
            }
            
            .skill-counter-text {
                font-size: 1.1rem;
                color: rgba(255,255,255,0.95);
                font-weight: 600;
                transition: all 0.4s ease;
                text-shadow: 0 1px 2px rgba(0,0,0,0.2);
            }
            
            .skill-counter-item:hover .skill-counter-text {
                color: #fff;
                transform: scale(1.05);
                text-shadow: 0 2px 4px rgba(0,0,0,0.3);
            }
            
            /* Responsive Design */
            @media (max-width: 768px) {
                .skill-counter-items {
                    flex-direction: column;
                }
                
                .skill-counter-item {
                    min-width: auto;
                }
                
                .skill-counter-main-title {
                    font-size: 2rem;
                }
                
                .skill-counter-number {
                    font-size: 2.5rem;
                }
            }
            
            /* Animation Classes */
            .skill-counter-item.animate {
                opacity: 1;
                transform: translateY(0);
            }
            
            .skill-counter-item.fadeInUp {
                animation: fadeInUp 0.6s ease forwards;
            }
            
            .skill-counter-item.fadeInDown {
                animation: fadeInDown 0.6s ease forwards;
            }
            
            .skill-counter-item.fadeInLeft {
                animation: fadeInLeft 0.6s ease forwards;
            }
            
            .skill-counter-item.fadeInRight {
                animation: fadeInRight 0.6s ease forwards;
            }
            
            .skill-counter-item.zoomIn {
                animation: zoomIn 0.6s ease forwards;
            }
            
            .skill-counter-item.slideInUp {
                animation: slideInUp 0.6s ease forwards;
            }
            
            .skill-counter-item.bounceIn {
                animation: bounceIn 0.6s ease forwards;
            }
            
            .skill-counter-item.flipInX {
                animation: flipInX 0.6s ease forwards;
            }
            
            /* Keyframe Animations */
            @keyframes fadeInUp {
                from {
                    opacity: 0;
                    transform: translateY(30px);
                }
                to {
                    opacity: 1;
                    transform: translateY(0);
                }
            }
            
            @keyframes fadeInDown {
                from {
                    opacity: 0;
                    transform: translateY(-30px);
                }
                to {
                    opacity: 1;
                    transform: translateY(0);
                }
            }
            
            @keyframes fadeInLeft {
                from {
                    opacity: 0;
                    transform: translateX(-30px);
                }
                to {
                    opacity: 1;
                    transform: translateX(0);
                }
            }
            
            @keyframes fadeInRight {
                from {
                    opacity: 0;
                    transform: translateX(30px);
                }
                to {
                    opacity: 1;
                    transform: translateX(0);
                }
            }
            
            @keyframes zoomIn {
                from {
                    opacity: 0;
                    transform: scale(0.3);
                }
                to {
                    opacity: 1;
                    transform: scale(1);
                }
            }
            
            @keyframes slideInUp {
                from {
                    opacity: 0;
                    transform: translateY(100px);
                }
                to {
                    opacity: 1;
                    transform: translateY(0);
                }
            }
            
            @keyframes bounceIn {
                0% {
                    opacity: 0;
                    transform: scale(0.3);
                }
                50% {
                    opacity: 1;
                    transform: scale(1.05);
                }
                70% {
                    transform: scale(0.9);
                }
                100% {
                    opacity: 1;
                    transform: scale(1);
                }
            }
            
            @keyframes flipInX {
                from {
                    opacity: 0;
                    transform: perspective(400px) rotateX(90deg);
                }
                40% {
                    transform: perspective(400px) rotateX(-20deg);
                }
                60% {
                    transform: perspective(400px) rotateX(10deg);
                }
                80% {
                    transform: perspective(400px) rotateX(-5deg);
                }
                to {
                    opacity: 1;
                    transform: perspective(400px) rotateX(0deg);
                }
            }
            
            /* Glass morphism effects */
            .skill-counter-item {
                backdrop-filter: blur(20px);
                -webkit-backdrop-filter: blur(20px);
            }
            
            /* Floating particles effect */
            .skill-counter-container::after {
                content: '';
                position: absolute;
                top: 0;
                left: 0;
                right: 0;
                bottom: 0;
                background-image: 
                    radial-gradient(circle at 20% 80%, rgba(255,255,255,0.15) 0%, transparent 50%),
                    radial-gradient(circle at 80% 20%, rgba(255,255,255,0.15) 0%, transparent 50%),
                    radial-gradient(circle at 40% 40%, rgba(255,255,255,0.08) 0%, transparent 50%),
                    radial-gradient(circle at 60% 60%, rgba(255,255,255,0.1) 0%, transparent 50%);
                animation: float 8s ease-in-out infinite;
                pointer-events: none;
            }
            
            @keyframes float {
                0%, 100% { transform: translateY(0px) rotate(0deg) scale(1); }
                25% { transform: translateY(-8px) rotate(0.5deg) scale(1.02); }
                50% { transform: translateY(8px) rotate(-0.5deg) scale(0.98); }
                75% { transform: translateY(-4px) rotate(0.3deg) scale(1.01); }
            }
            
            /* Electric effect */
            .skill-counter-item:hover {
                animation: electricPulse 0.8s ease;
            }
            
            @keyframes electricPulse {
                0% {
                    box-shadow: 0 20px 40px rgba(0, 0, 0, 0.2);
                }
                25% {
                    box-shadow: 0 20px 40px rgba(0, 0, 0, 0.2), 0 0 20px rgba(255, 255, 255, 0.6);
                }
                50% {
                    box-shadow: 0 20px 40px rgba(0, 0, 0, 0.2), 0 0 30px rgba(255, 255, 255, 0.8);
                }
                75% {
                    box-shadow: 0 20px 40px rgba(0, 0, 0, 0.2), 0 0 20px rgba(255, 255, 255, 0.6);
                }
                100% {
                    box-shadow: 0 20px 40px rgba(0, 0, 0, 0.2);
                }
            }
            
            /* Sparkle effect */
            .skill-counter-item::before {
                content: '';
                position: absolute;
                top: 0;
                left: -100%;
                width: 100%;
                height: 100%;
                background: linear-gradient(90deg, transparent, rgba(255,255,255,0.4), transparent);
                transition: left 0.5s ease;
                z-index: 1;
            }
            
            .skill-counter-item:hover::before {
                left: 100%;
                animation: sparkle 0.6s ease;
            }
            
            @keyframes sparkle {
                0% { opacity: 0; }
                50% { opacity: 1; }
                100% { opacity: 0; }
            }
            
            /* Modern glow and shine effects */
            .skill-counter-item::before {
                content: '';
                position: absolute;
                top: 0;
                left: -100%;
                width: 100%;
                height: 100%;
                background: linear-gradient(90deg, transparent, rgba(255, 255, 255, 0.2), transparent);
                transition: left 0.6s ease;
                z-index: 1;
            }
            
            .skill-counter-item:hover::before {
                left: 100%;
            }
            
            .skill-counter-item::after {
                content: '';
                position: absolute;
                top: 50%;
                left: 50%;
                width: 0;
                height: 0;
                background: radial-gradient(circle, rgba(255, 255, 255, 0.3) 0%, transparent 70%);
                border-radius: 50%;
                transform: translate(-50%, -50%);
                transition: all 0.4s ease;
                z-index: 0;
            }
            
            .skill-counter-item:hover::after {
                width: 200px;
                height: 200px;
            }
            
            /* Neon glow for numbers on hover */
                        .skill-counter-item:hover .skill-counter-number {
                text-shadow:
                    0 0 10px rgba(255,255,255,0.9),
                    0 0 20px rgba(255,255,255,0.7),
                    0 0 30px rgba(255,255,255,0.5),
                    0 0 40px rgba(255,255,255,0.3);
                color: #ffffff;
            }
            
            /* Icon glow effect */
            .skill-counter-item:hover .skill-counter-icon i,
            .skill-counter-item:hover .skill-counter-icon svg {
                text-shadow:
                    0 0 15px rgba(255,255,255,0.9),
                    0 0 30px rgba(255,255,255,0.7),
                    0 0 45px rgba(255,255,255,0.5),
                    0 0 60px rgba(255,255,255,0.3);
                transform: scale(1.15) rotate(8deg);
            }
            
            /* Modern pulse effect */
            .skill-counter-item:hover::after {
                animation: modernPulse 1.2s ease-in-out;
            }
            
            @keyframes modernPulse {
                0% { opacity: 0; transform: scale(0.8); }
                50% { opacity: 1; transform: scale(1.1); }
                100% { opacity: 0; transform: scale(1.2); }
            }
            
            /* Floating animation for cards */
            .skill-counter-item {
                animation: float 6s ease-in-out infinite;
            }
            
            .skill-counter-item:nth-child(2) {
                animation-delay: 0.5s;
            }
            
            .skill-counter-item:nth-child(3) {
                animation-delay: 1s;
            }
            
            .skill-counter-item:nth-child(4) {
                animation-delay: 1.5s;
            }
            
            @keyframes float {
                0%, 100% { transform: translateY(0px) rotate(0deg); }
                25% { transform: translateY(-5px) rotate(0.5deg); }
                50% { transform: translateY(5px) rotate(-0.5deg); }
                75% { transform: translateY(-3px) rotate(0.3deg); }
            }
        </style>

        <script>
        (function() {
            'use strict';
            
            // Counter animation function
            function animateCounter(element, target, duration) {
                const start = 0;
                const increment = target / (duration * 60); // 60 FPS
                let current = start;
                
                const timer = setInterval(() => {
                    current += increment;
                    if (current >= target) {
                        current = target;
                        clearInterval(timer);
                    }
                    element.textContent = Math.floor(current);
                }, 1000 / 60);
            }
            
            // Intersection Observer for triggering animations
            function initCounterAnimations() {
                const counterItems = document.querySelectorAll('#<?php echo esc_js($widget_id); ?> .skill-counter-item');
                
                const observer = new IntersectionObserver((entries) => {
                    entries.forEach((entry, index) => {
                        if (entry.isIntersecting) {
                            const item = entry.target;
                            const delay = parseFloat(item.dataset.delay) || 0;
                            const animationType = item.dataset.animation || 'fadeInUp';
                            const animationDuration = parseFloat(item.dataset.animationDuration) || 1;
                            
                            // Apply animation after delay
                            setTimeout(() => {
                                item.style.animationDuration = animationDuration + 's';
                                item.classList.add('animate', animationType);
                                
                                // Start counter animation if enabled
                                <?php if ($enable_counter_animation === 'yes') : ?>
                                const counterElement = item.querySelector('.skill-counter-number');
                                const target = parseInt(item.dataset.number);
                                const counterDuration = parseFloat(item.dataset.duration) || 2;
                                
                                setTimeout(() => {
                                    animateCounter(counterElement, target, counterDuration);
                                }, animationDuration * 1000);
                                <?php endif; ?>
                                
                            }, delay * 1000);
                            
                            observer.unobserve(item);
                        }
                    });
                }, {
                    threshold: 0.1,
                    rootMargin: '0px 0px -50px 0px'
                });
                
                counterItems.forEach(item => {
                    observer.observe(item);
                });
            }
            
            // Initialize when DOM is ready
            if (document.readyState === 'loading') {
                document.addEventListener('DOMContentLoaded', initCounterAnimations);
            } else {
                initCounterAnimations();
            }
            
            // Re-initialize for Elementor editor
            if (typeof elementorFrontend !== 'undefined') {
                elementorFrontend.hooks.addAction('frontend/element_ready/skill_counter_animation.default', function() {
                    setTimeout(initCounterAnimations, 100);
                });
            }
        })();
        </script>
        <?php
    }
}
