<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Repeater;

class Skill_Educational_Widget extends Widget_Base {

    // Widget slug (must be unique)
    public function get_name() {
        return 'skill_educational_modern';
    }

    // Widget title shown in Elementor panel (Persian)
    public function get_title() {
        return __( 'سابقه تحصیلی مدرن', 'skill' );
    }

    // Elementor icon
    public function get_icon() {
        return 'eicon-time-line';
    }

    // Category in Elementor panel
    public function get_categories() {
        return [ 'skill-category' ];
    }

    // Controls registration (all in Persian labels, English comments)
    protected function register_controls() {
        

        // Title + Description controls (mirroring skill-title-desc-widget)
        $this->start_controls_section('tdw_content', [ 'label' => __( 'محتوا', 'skill' ), 'tab' => Controls_Manager::TAB_CONTENT ]);
        $this->add_control('show_title', [ 'label' => __( 'نمایش عنوان', 'skill' ), 'type' => Controls_Manager::SWITCHER, 'label_on' => __( 'بله', 'skill' ), 'label_off' => __( 'خیر', 'skill' ), 'return_value' => 'yes', 'default' => 'yes' ]);
        $this->add_control('title_text', [ 'label' => __( 'عنوان', 'skill' ), 'type' => Controls_Manager::TEXT, 'default' => __( 'سابقه تحصیلی', 'skill' ), 'condition' => [ 'show_title' => 'yes' ] ]);
        $this->add_control('show_dot', [ 'label' => __( 'نمایش مربع آبی', 'skill' ), 'type' => Controls_Manager::SWITCHER, 'label_on' => __( 'بله', 'skill' ), 'label_off' => __( 'خیر', 'skill' ), 'return_value' => 'yes', 'default' => 'yes' ]);
        $this->add_control('dot_position', [ 'label' => __( 'جایگاه مربع', 'skill' ), 'type' => Controls_Manager::SELECT, 'options' => [ 'right' => __( 'راست عنوان', 'skill' ), 'left' => __( 'چپ عنوان', 'skill' ) ], 'default' => 'right', 'condition' => [ 'show_dot' => 'yes' ] ]);
        $this->add_control('show_desc', [ 'label' => __( 'نمایش توضیح کوتاه', 'skill' ), 'type' => Controls_Manager::SWITCHER, 'label_on' => __( 'بله', 'skill' ), 'label_off' => __( 'خیر', 'skill' ), 'return_value' => 'yes', 'default' => 'yes' ]);
        $this->add_control('desc_text', [ 'label' => __( 'توضیح کوتاه', 'skill' ), 'type' => Controls_Manager::TEXTAREA, 'default' => __( 'توضیح کوتاه درباره سوابق تحصیلی', 'skill' ), 'rows' => 2, 'condition' => [ 'show_desc' => 'yes' ] ]);
        $this->end_controls_section();

        // Repeater: educational items
        $this->start_controls_section(
            'section_educational_items',
            [
                'label' => __( 'آیتم‌ها', 'skill' ),
            ]
        );

        $repeater = new Repeater();

        // Date range (e.g., 1400–1404)
        $repeater->add_control(
            'date_range',
            [
                'label' => __( 'بازه زمانی', 'skill' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'از 1400 تا 1404', 'skill' ),
                'label_block' => true,
            ]
        );

        // University title/name
        $repeater->add_control(
            'university',
            [
                'label' => __( 'نام دانشگاه/موسسه', 'skill' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'دانشگاه پیام نور شهرکرد', 'skill' ),
                'label_block' => true,
            ]
        );

        // Degree level
        $repeater->add_control(
            'degree',
            [
                'label' => __( 'مقطع', 'skill' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'فوق لیسانس', 'skill' ),
                'label_block' => true,
            ]
        );

        // Major/field of study
        $repeater->add_control(
            'major',
            [
                'label' => __( 'رشته', 'skill' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'مهندسی کامپیوتر', 'skill' ),
                'label_block' => true,
            ]
        );

        // Logo image
        $repeater->add_control(
            'logo',
            [
                'label' => __( 'لوگو/تصویر', 'skill' ),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => \Elementor\Utils::get_placeholder_image_src(),
                ],
            ]
        );

        // Short description
        $repeater->add_control(
            'description',
            [
                'label' => __( 'توضیحات کوتاه', 'skill' ),
                'type' => Controls_Manager::TEXTAREA,
                'default' => __( 'فارغ‌التحصیل با تخصص در توسعه وب و امنیت.', 'skill' ),
                'rows' => 3,
            ]
        );

        $this->add_control(
            'items',
            [
                'label' => __( 'آیتم‌ها', 'skill' ),
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'date_range' => __( 'از 1400 تا 1404', 'skill' ),
                        'university' => __( 'دانشگاه پیام نور شهرکرد', 'skill' ),
                        'degree' => __( 'فوق لیسانس', 'skill' ),
                        'major' => __( 'مهندسی کامپیوتر', 'skill' ),
                        'description' => __( 'تمرکز بر توسعه وب، امنیت و الگوریتم‌ها.', 'skill' ),
                    ],
                ],
                'title_field' => '{{{ university }}} – {{{ degree }}}',
            ]
        );

        $this->end_controls_section();

        // Style section for educational items
        $this->start_controls_section(
            'section_style',
            [
                'label' => __( 'استایل', 'skill' ),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        // Colors and sizes via selectors
        $this->add_control(
            'date_color',
            [
                'label' => __( 'رنگ تاریخ', 'skill' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#607d8b',
                'selectors' => [
                    '{{WRAPPER}} .skill_educational-date span' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'university_color',
            [
                'label' => __( 'رنگ عنوان دانشگاه', 'skill' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#1976d2',
                'selectors' => [
                    '{{WRAPPER}} .skill_educational-meta h2' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'label_color',
            [
                'label' => __( 'رنگ برچسب‌ها', 'skill' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#333333',
                'selectors' => [
                    '{{WRAPPER}} .skill_educational-meta span' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'text_color',
            [
                'label' => __( 'رنگ متن توضیحات', 'skill' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#607d8b',
                'selectors' => [
                    '{{WRAPPER}} .skill_educational-meta p, {{WRAPPER}} .skill_educational-description p' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'item_border_color',
            [
                'label' => __( 'رنگ حاشیه آیتم', 'skill' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#B6CEFF',
                'selectors' => [
                    '{{WRAPPER}} .skill_educational-iteme' => 'border-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'item_hover_border_color',
            [
                'label' => __( 'رنگ حاشیه هاور', 'skill' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#571F8B',
                'selectors' => [
                    '{{WRAPPER}} .skill_educational-items:hover .skill_educational-iteme' => 'border-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'item_bg_color',
            [
                'label' => __( 'پس‌زمینه آیتم', 'skill' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#ffffff',
                'selectors' => [
                    '{{WRAPPER}} .skill_educational-iteme' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'logo_size',
            [
                'label' => __( 'اندازه لوگو', 'skill' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px' ],
                'range' => [
                    'px' => [ 'min' => 40, 'max' => 120 ],
                ],
                'default' => [ 'size' => 60, 'unit' => 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .skill_educational-info img' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Style section for Title + Desc
        $this->start_controls_section('tdw_style', [ 'label' => __( 'استایل عنوان/توضیح', 'skill' ), 'tab' => Controls_Manager::TAB_STYLE ]);
        $this->add_responsive_control('tdw_align', [ 'label' => __( 'چینش', 'skill' ), 'type' => Controls_Manager::CHOOSE, 'options' => [ 'right' => [ 'title' => __( 'راست', 'skill' ), 'icon' => 'eicon-text-align-right' ], 'center' => [ 'title' => __( 'وسط', 'skill' ), 'icon' => 'eicon-text-align-center' ], 'left' => [ 'title' => __( 'چپ', 'skill' ), 'icon' => 'eicon-text-align-left' ] ], 'default' => 'right', 'selectors' => [ '{{WRAPPER}} .tdw-wrap' => 'text-align: {{VALUE}};' ] ]);
        $this->add_control('tdw_title_color', [ 'label' => __( 'رنگ عنوان', 'skill' ), 'type' => Controls_Manager::COLOR, 'default' => '#1C1C1C', 'selectors' => [ '{{WRAPPER}} .tdw-title' => 'color: {{VALUE}};' ] ]);
        $this->add_group_control(\Elementor\Group_Control_Typography::get_type(), [ 'name' => 'tdw_title_typo', 'label' => __( 'تایپوگرافی عنوان', 'skill' ), 'selector' => '{{WRAPPER}} .tdw-title' ]);
        $this->add_control('tdw_desc_color', [ 'label' => __( 'رنگ توضیح', 'skill' ), 'type' => Controls_Manager::COLOR, 'default' => '#607d8b', 'selectors' => [ '{{WRAPPER}} .tdw-desc' => 'color: {{VALUE}};' ] ]);
        $this->add_group_control(\Elementor\Group_Control_Typography::get_type(), [ 'name' => 'tdw_desc_typo', 'label' => __( 'تایپوگرافی توضیح', 'skill' ), 'selector' => '{{WRAPPER}} .tdw-desc' ]);
        $this->add_control('tdw_dot_color', [ 'label' => __( 'رنگ مربع', 'skill' ), 'type' => Controls_Manager::COLOR, 'default' => '#1e73be', 'selectors' => [ '{{WRAPPER}} .tdw-dot' => 'background-color: {{VALUE}};' ] ]);
        $this->add_control('tdw_dot_size', [ 'label' => __( 'سایز مربع (px)', 'skill' ), 'type' => Controls_Manager::SLIDER, 'size_units' => [ 'px' ], 'range' => [ 'px' => [ 'min' => 6, 'max' => 24 ] ], 'default' => [ 'unit' => 'px', 'size' => 10 ], 'selectors' => [ '{{WRAPPER}} .tdw-dot' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};' ] ]);
        $this->end_controls_section();
    }

    // Render widget output
    protected function render() {
        $settings = $this->get_settings_for_display();

        $dot_right = ( $settings['dot_position'] ?? 'right' ) === 'right';
        ?>
        <div class="skill_educational-modern" style="direction: rtl; text-align: right;">
            <div class="tdw-wrap">
                <div class="tdw-title-row" style="display:inline-flex; align-items:center; gap:8px; <?php echo $dot_right ? 'flex-direction: row-reverse;' : ''; ?>">
                    <?php if ( ! empty( $settings['show_title'] ) && $settings['show_title'] === 'yes' ) : ?>
                        <h2 class="tdw-title"><?php echo esc_html( $settings['title_text'] ?: '' ); ?></h2>
                    <?php endif; ?>
                    <?php if ( ! empty( $settings['show_dot'] ) && $settings['show_dot'] === 'yes' ) : ?>
                        <span class="tdw-dot" aria-hidden="true" style="display:inline-block; border-radius:3px;"></span>
                    <?php endif; ?>
                </div>
                <?php if ( ! empty( $settings['show_desc'] ) && $settings['show_desc'] === 'yes' && ! empty( $settings['desc_text'] ) ) : ?>
                    <div class="tdw-desc" style="margin-top:10px;">
                        <?php echo esc_html( $settings['desc_text'] ); ?>
                    </div>
                <?php endif; ?>
            </div>

            <div class="skill_educational-items">
                <?php if ( ! empty( $settings['items'] ) ) : ?>
                    <?php foreach ( $settings['items'] as $item ) : ?>
                        <div class="skill_educational-iteme" data-date="<?php echo esc_attr( $item['date_range'] ); ?>">

                            <div class="skill_educational-info">
                                <?php if ( ! empty( $item['logo']['url'] ) ) : ?>
                                    <img src="<?php echo esc_url( $item['logo']['url'] ); ?>" alt="<?php echo esc_attr( $item['university'] ); ?>">
                                <?php endif; ?>
                                <div class="skill_educational-meta">
                                    <h2><?php echo esc_html( $item['university'] ); ?></h2>
                                    <p>
                                        <span><?php echo esc_html__( 'مقطع:', 'skill' ); ?></span>
                                        <?php echo esc_html( $item['degree'] ); ?>
                                    </p>
                                    <p>
                                        <span><?php echo esc_html__( 'رشته:', 'skill' ); ?></span>
                                        <?php echo esc_html( $item['major'] ); ?>
                                    </p>
                                </div>
                            </div>

                            <?php if ( ! empty( $item['description'] ) ) : ?>
                                <div class="skill_educational-description">
                                    <p><?php echo esc_html( $item['description'] ); ?></p>
                                </div>
                            <?php endif; ?>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
                <div class="skill_educational-date-float"><span></span></div>
            </div>
        </div>

        <style>
        /* Vertical line (optional) to enhance look */
        .skill_educational-items { position: relative; padding-right: 72px;margin-top: 25px; }
        .skill_educational-items:before { content: ""; position: absolute; top: 0; bottom: 0; right: 28px; width: 8px; background: linear-gradient(180deg,#90caf9 0%,#1976d2 100%); border-radius: 8px; }
        /* Single moving dot on the timeline */

        /* Floating date outside cards */
        .skill_educational-date-float { position: absolute; right: 8px; top: var(--date-top, 30px); transform: translateY(-50%); z-index: 2; pointer-events: none; opacity: var(--date-opacity, 0); transition: top .2s ease, opacity .15s ease; }
        .skill_educational-date-float span { display: inline-block; font-size: 13px; color: #607d8b; font-weight: 700; background: rgba(255,255,255,0.95); padding: 4px 8px; border-radius: 6px; box-shadow: 0 2px 8px rgba(0,0,0,0.1); white-space: nowrap; }
        .skill_educational-date-float:after { content: ""; position: absolute; right: -8px; top: 50%; transform: translateY(-50%); width: 0; height: 0; border-top: 6px solid transparent; border-bottom: 6px solid transparent; border-right: 8px solid #1976d2; filter: drop-shadow(0 0 2px rgba(25,118,210,.35)); }

        /* Card base - collapsed by default to hide description */
        .skill_educational-iteme { position: relative; z-index: 1; border: 1px solid #B6CEFF; border-radius: 16px; padding: 14px; cursor: pointer; background: #ffffff; overflow: hidden; transition: box-shadow .25s ease, transform .25s ease, margin-right .25s ease, height .25s ease; height: 100px; margin-bottom: 22px; }

        /* Hover: move card left and increase height to reveal description and date */
        .skill_educational-iteme:hover { border-color: #571F8B; box-shadow: 0 12px 30px rgba(87,31,139,0.22), 0 4px 12px rgba(25,118,210,0.16); transform: translateY(-2px); margin-right: 40px; height: 140px; }
        .skill_educational-iteme:hover:before { content: ""; position: absolute; inset: 0; border-radius: 16px; pointer-events: none; box-shadow: inset 0 0 0 2px rgba(123,31,162,.12), inset 0 0 0 1px rgba(33,99,242,.15); }

        .skill_educational-info { display: flex; align-items: center; }
        .skill_educational-info img { object-fit: contain; border-radius: 12px; background: #e3f2fd; display: flex; align-items: center; justify-content: center; overflow: hidden; box-shadow: 0 2px 8px 0 rgba(25, 118, 210, 0.08); border: 2px solid #90caf9; align-self: flex-start; width: 60px; height: 60px; margin-left: 15px; }

        .skill_educational-meta h2 { font-size: 15px; font-weight: 800; color: #1976d2; margin: 0 0 6px 0; }
        .skill_educational-meta span { font-size: 16px; font-weight: 700; color: #333333; }
        .skill_educational-meta p { font-size: 14px; color: #607d8b; margin: 0; }

        .skill_educational-description { margin-top: 12px; }
        .skill_educational-description p { font-size: 14px; color: #607d8b; margin: 0 0 10px 0; }
        </style>
        <style>
            @media (max-width: 768px) {
            .skill_educational-iteme:hover {
                height: 185px;
            }
        }

        @media (max-width: 425px) {
            /* Switch to horizontal timeline */
            .skill_educational-items { padding-right: 0; padding-top: 56px; }
            .skill_educational-items:before { right: 0; left: 0; width: 100%; height: 8px; top: 28px; bottom: auto; }
            /* Hide vertical moving dot on mobile */

            /* Hover behavior: push down and expand height */
            .skill_educational-iteme { height: 100px; margin-bottom: 16px; }
            .skill_educational-iteme:hover { margin-right: 0; margin-top: 24px; height: 170px; }

            /* Date floats above the card centered horizontally */
            .skill_educational-date-float { right: auto; left: var(--date-left, 50%); top: 10px; transform: translate(-50%, -50%); }
        }

        @media (max-width: 375px) {
            .skill_educational-iteme:hover {
                height: 200px;
            }
        }

        @media (max-width: 320px) {
            .skill_educational-iteme:hover {
                height: 250px;
            }
        }
        </style>
        <script>
        (function(){
            // Scope to this widget container only
            var container = document.currentScript.previousElementSibling.previousElementSibling.previousElementSibling;
            if(!container || !container.querySelector) return;
            var list = container.querySelector('.skill_educational-items');
            if(!list) return;
            var items = list.querySelectorAll('.skill_educational-iteme');
            var dateFloat = list.querySelector('.skill_educational-date-float span');
            function moveDotTo(el){
                var rect = el.getBoundingClientRect();
                var listRect = list.getBoundingClientRect();
                var centerY = rect.top + rect.height/2 - listRect.top;
                var centerX = rect.left + rect.width/2 - listRect.left;
                // For desktop (vertical)
                list.style.setProperty('--dot-top', centerY + 'px');
                list.style.setProperty('--dot-opacity', 1);
                list.style.setProperty('--date-top', centerY + 'px');
                list.style.setProperty('--date-opacity', 1);
                // For mobile (horizontal)
                list.style.setProperty('--date-left', centerX + 'px');
                if(dateFloat){ dateFloat.textContent = el.getAttribute('data-date') || ''; }
            }
            items.forEach(function(it){
                it.addEventListener('mouseenter', function(){ moveDotTo(it); });
            });
            // Initialize to first item if exists
            if(items[0]){ moveDotTo(items[0]); }
        })();
        </script>
        <?php
    }
}
