<?php
/**
 * Skill Panel Pricing Widget
 * 
 * @package Skill
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

/**
 * Skill Panel Pricing Widget Class
 */
class Skill_Panel_Pricing_Widget extends \Elementor\Widget_Base {

    /**
     * Get widget name
     */
    public function get_name() {
        return 'skill-panel-pricing';
    }

    /**
     * Get widget title
     */
    public function get_title() {
        return esc_html__('پنل قیمت گذاری', 'skill');
    }

    /**
     * Get widget icon
     */
    public function get_icon() {
        return 'eicon-price-table';
    }

    /**
     * Get widget categories
     */
    public function get_categories() {
        return ['skill-category'];
    }

    /**
     * Get widget keywords
     */
    public function get_keywords() {
        return ['pricing', 'panel', 'price', 'table', 'skill'];
    }

    /**
     * Get widget dependencies
     */
    public function get_script_depends() {
        return [];
    }

    /**
     * Get widget style dependencies
     */
    public function get_style_depends() {
        return ['skill-panel-pricing-widget'];
    }

    /**
     * Register widget controls
     */
    protected function register_controls() {
        
        // Content Section
        $this->start_controls_section(
            'content_section',
            [
                'label' => esc_html__('محتوای پنل', 'skill'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $repeater = new \Elementor\Repeater();

        $repeater->add_control(
            'panel_image',
            [
                'label' => esc_html__('تصویر پنل', 'skill'),
                'type' => \Elementor\Controls_Manager::MEDIA,
                'default' => [
                    'url' => \Elementor\Utils::get_placeholder_image_src(),
                ],
            ]
        );

        $repeater->add_control(
            'panel_title',
            [
                'label' => esc_html__('عنوان پنل', 'skill'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => esc_html__('پلن وی آی پی', 'skill'),
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'original_price',
            [
                'label' => esc_html__('قیمت اصلی', 'skill'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => '15,000,000 تومان',
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'discounted_price',
            [
                'label' => esc_html__('قیمت با تخفیف', 'skill'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => '14,000,000 تومان',
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'features_list',
            [
                'label' => esc_html__('ویژگی‌ها', 'skill'),
                'type' => \Elementor\Controls_Manager::TEXTAREA,
                'default' => "طراحی اختصاصی\nطراحی بدون قالب آماده\nتخفیف هاست و دامنه",
                'description' => esc_html__('هر ویژگی را در یک خط جداگانه بنویسید', 'skill'),
            ]
        );

        $repeater->add_control(
            'button_text',
            [
                'label' => esc_html__('متن دکمه', 'skill'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => esc_html__('سفارش طرح', 'skill'),
            ]
        );

        $repeater->add_control(
            'button_link',
            [
                'label' => esc_html__('لینک دکمه', 'skill'),
                'type' => \Elementor\Controls_Manager::URL,
                'placeholder' => esc_html__('https://your-link.com', 'skill'),
                'default' => [
                    'url' => '#',
                    'is_external' => false,
                    'nofollow' => false,
                ],
            ]
        );

        $repeater->add_control(
            'panel_image_background_color',
            [
                'label' => esc_html__('رنگ پس زمینه تصویر', 'skill'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#f8f9fa',
            ]
        );



        $this->add_control(
            'pricing_panels',
            [
                'label' => esc_html__('پنل‌های قیمت گذاری', 'skill'),
                'type' => \Elementor\Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'panel_title' => esc_html__('پلن وی آی پی', 'skill'),
                        'original_price' => '15,000,000 تومان',
                        'discounted_price' => '14,000,000 تومان',
                        'button_text' => esc_html__('سفارش طرح', 'skill'),
                    ],
                ],
                'title_field' => '{{{ panel_title }}}',
            ]
        );

        $this->end_controls_section();

        // Style Section - Panel
        $this->start_controls_section(
            'panel_style_section',
            [
                'label' => esc_html__('استایل پنل', 'skill'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name' => 'panel_border',
                'label' => esc_html__('حاشیه', 'skill'),
                'selector' => '{{WRAPPER}} .skill_box-panel-pricing',
            ]
        );

        $this->add_control(
            'panel_border_color',
            [
                'label' => esc_html__('رنگ حاشیه', 'skill'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .skill_box-panel-pricing' => 'border-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'panel_border_radius',
            [
                'label' => esc_html__('شعاع گوشه', 'skill'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .skill_box-panel-pricing' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'default' => [
                    'top' => '16',
                    'right' => '16',
                    'bottom' => '16',
                    'left' => '16',
                    'unit' => 'px',
                ],
            ]
        );

        $this->add_control(
            'panel_background_color',
            [
                'label' => esc_html__('رنگ پس زمینه', 'skill'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .skill_box-panel-pricing' => 'background-color: {{VALUE}};',
                ],
            ]
        );



        $this->end_controls_section();

        // Style Section - Title
        $this->start_controls_section(
            'title_style_section',
            [
                'label' => esc_html__('استایل عنوان', 'skill'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'title_typography',
                'label' => esc_html__('تایپوگرافی', 'skill'),
                'selector' => '{{WRAPPER}} .skill_pric-title h3',
            ]
        );

        $this->add_control(
            'title_color',
            [
                'label' => esc_html__('رنگ متن', 'skill'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .skill_pric-title h3' => 'color: {{VALUE}};',
                ],
                'default' => '#7B7875',
            ]
        );

        $this->end_controls_section();

        // Style Section - Prices
        $this->start_controls_section(
            'price_style_section',
            [
                'label' => esc_html__('استایل قیمت‌ها', 'skill'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'original_price_typography',
                'label' => esc_html__('تایپوگرافی قیمت اصلی', 'skill'),
                'selector' => '{{WRAPPER}} .skill_pric-del',
            ]
        );

        $this->add_control(
            'original_price_color',
            [
                'label' => esc_html__('رنگ قیمت اصلی', 'skill'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .skill_pric-del' => 'color: {{VALUE}};',
                ],
                'default' => '#403d39',
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'discounted_price_typography',
                'label' => esc_html__('تایپوگرافی قیمت تخفیف', 'skill'),
                'selector' => '{{WRAPPER}} .skill_pric-price',
            ]
        );

        $this->add_control(
            'discounted_price_color',
            [
                'label' => esc_html__('رنگ قیمت تخفیف', 'skill'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .skill_pric-price' => 'color: {{VALUE}};',
                ],
                'default' => '#403d39',
            ]
        );

        $this->end_controls_section();

        // Style Section - Features
        $this->start_controls_section(
            'features_style_section',
            [
                'label' => esc_html__('استایل ویژگی‌ها', 'skill'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'features_typography',
                'label' => esc_html__('تایپوگرافی', 'skill'),
                'selector' => '{{WRAPPER}} .skill_pric-list-daata ul li',
            ]
        );

        $this->add_control(
            'features_color',
            [
                'label' => esc_html__('رنگ متن', 'skill'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .skill_pric-list-daata ul li' => 'color: {{VALUE}};',
                ],
                'default' => '#4D4E50',
            ]
        );

        $this->add_control(
            'icon_color',
            [
                'label' => esc_html__('رنگ آیکون', 'skill'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .skill_pric-list-daata ul li i' => 'color: {{VALUE}};',
                ],
                'default' => '#2575fc',
            ]
        );

        $this->end_controls_section();

        // Style Section - Button
        $this->start_controls_section(
            'button_style_section',
            [
                'label' => esc_html__('استایل دکمه', 'skill'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'button_typography',
                'label' => esc_html__('تایپوگرافی', 'skill'),
                'selector' => '{{WRAPPER}} .skill_pric-btn',
            ]
        );

        $this->start_controls_tabs('button_tabs');

        $this->start_controls_tab(
            'button_normal_tab',
            [
                'label' => esc_html__('عادی', 'skill'),
            ]
        );

        $this->add_control(
            'button_text_color',
            [
                'label' => esc_html__('رنگ متن', 'skill'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .skill_pric-btn' => 'color: {{VALUE}};',
                ],
                'default' => '#ffffff',
            ]
        );

        $this->add_control(
            'button_background_color',
            [
                'label' => esc_html__('رنگ پس زمینه', 'skill'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .skill_pric-btn' => 'background-color: {{VALUE}};',
                ],
                'default' => '#2575fc',
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'button_hover_tab',
            [
                'label' => esc_html__('هاور', 'skill'),
            ]
        );

        $this->add_control(
            'button_hover_text_color',
            [
                'label' => esc_html__('رنگ متن', 'skill'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .skill_pric-btn:hover' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'button_hover_background_color',
            [
                'label' => esc_html__('رنگ پس زمینه', 'skill'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .skill_pric-btn:hover' => 'background-color: {{VALUE}};',
                ],
                'default' => '#223a5f',
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->add_control(
            'button_border_radius',
            [
                'label' => esc_html__('شعاع گوشه', 'skill'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .skill_pric-btn' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'default' => [
                    'top' => '50',
                    'right' => '50',
                    'bottom' => '50',
                    'left' => '50',
                    'unit' => 'px',
                ],
            ]
        );

        $this->add_responsive_control(
            'button_padding',
            [
                'label' => esc_html__('فاصله داخلی', 'skill'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .skill_pric-btn' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'default' => [
                    'top' => '10',
                    'right' => '25',
                    'bottom' => '10',
                    'left' => '25',
                    'unit' => 'px',
                ],
            ]
        );

        $this->end_controls_section();

        // Style Section - Progress Bar
        $this->start_controls_section(
            'progress_bar_style_section',
            [
                'label' => esc_html__('استایل نوار پیشرفت', 'skill'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'progress_bar_color',
            [
                'label' => esc_html__('رنگ نوار', 'skill'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .bar' => 'background-color: {{VALUE}};',
                ],
                'default' => '#2575fc26',
            ]
        );

        $this->add_control(
            'progress_bar_height',
            [
                'label' => esc_html__('ارتفاع', 'skill'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'range' => [
                    'px' => [
                        'min' => 1,
                        'max' => 50,
                        'step' => 1,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 8,
                ],
                'selectors' => [
                    '{{WRAPPER}} .bar' => 'height: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'progress_bar_border_radius',
            [
                'label' => esc_html__('شعاع گوشه', 'skill'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .bar' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'default' => [
                    'top' => '0',
                    'right' => '0',
                    'bottom' => '16',
                    'left' => '16',
                    'unit' => 'px',
                ],
            ]
        );

        $this->end_controls_section();
    }

    /**
     * Render widget output on the frontend
     */
    protected function render() {
        $settings = $this->get_settings_for_display();

        if (empty($settings['pricing_panels'])) {
            return;
        }

        echo '<div class="container">';
        
        foreach ($settings['pricing_panels'] as $panel) {
            $this->render_panel($panel);
        }
        
        echo '</div>';
    }

    /**
     * Render individual pricing panel
     */
    private function render_panel($panel) {
        $image_url = $panel['panel_image']['url'] ?? '';
        $title = $panel['panel_title'] ?? '';
        $original_price = $panel['original_price'] ?? '';
        $discounted_price = $panel['discounted_price'] ?? '';
        $features = $panel['features_list'] ?? '';
        $button_text = $panel['button_text'] ?? '';
        $button_link = $panel['button_link']['url'] ?? '#';
        $button_target = $panel['button_link']['is_external'] ? ' target="_blank"' : '';
        $button_nofollow = $panel['button_link']['nofollow'] ? ' rel="nofollow"' : '';
        $image_bg_color = $panel['panel_image_background_color'] ?? '#f8f9fa';

        // Convert features text to array
        $features_array = array_filter(array_map('trim', explode("\n", $features)));

        ?>
        <div class="skill_panel-pricing">
            <div class="skill_box-panel-pricing">
                <div class="skill_panel-images" style="background-color: <?php echo esc_attr($image_bg_color); ?>;">
                    <?php if ($image_url): ?>
                        <img src="<?php echo esc_url($image_url); ?>" alt="<?php echo esc_attr($title); ?>">
                    <?php endif; ?>
                </div>
                <div class="skill_panel-pricing-content">
                    <div class="skill_pric-title">
                        <div>
                            <?php if ($title): ?>
                                <h3><?php echo esc_html($title); ?></h3>
                            <?php endif; ?>
                            <?php if ($original_price): ?>
                                <span class="skill_pric-del"><?php echo esc_html($original_price); ?></span>
                            <?php endif; ?>
                            <?php if ($discounted_price): ?>
                                <span class="skill_pric-price"><?php echo esc_html($discounted_price); ?></span>
                            <?php endif; ?>
                        </div>
                    </div>
                    <div class="skill_pric-list-daata">
                        <?php if (!empty($features_array)): ?>
                            <ul>
                                <?php foreach ($features_array as $feature): ?>
                                    <li><i class="fas fa-link"></i><?php echo esc_html($feature); ?></li>
                                <?php endforeach; ?>
                            </ul>
                        <?php endif; ?>
                    </div>
                    <div class="skill_pric-button">
                        <?php if ($button_text): ?>
                            <a href="<?php echo esc_url($button_link); ?>" class="skill_pric-btn"<?php echo $button_target . $button_nofollow; ?>>
                                <?php echo esc_html($button_text); ?>
                            </a>
                        <?php endif; ?>
                    </div>
                </div>
                <div class="bar"></div>
            </div>
        </div>
        <?php
    }

    /**
     * Render widget output in the editor
     */
    protected function content_template() {
        ?>
        <# if (settings.pricing_panels && settings.pricing_panels.length > 0) { #>
            <div class="container">
                <# _.each(settings.pricing_panels, function(panel) { #>
                    <div class="skill_panel-pricing">
                        <div class="skill_box-panel-pricing">
                            <div class="skill_panel-images" style="background-color: {{ panel.panel_image_background_color || '#f8f9fa' }};">
                                <# if (panel.panel_image && panel.panel_image.url) { #>
                                    <img src="{{{ panel.panel_image.url }}}" alt="{{{ panel.panel_title }}}">
                                <# } #>
                            </div>
                            <div class="skill_panel-pricing-content">
                                <div class="skill_pric-title">
                                    <div>
                                        <# if (panel.panel_title) { #>
                                            <h3>{{{ panel.panel_title }}}</h3>
                                        <# } #>
                                        <# if (panel.original_price) { #>
                                            <span class="skill_pric-del">{{{ panel.original_price }}}</span>
                                        <# } #>
                                        <# if (panel.discounted_price) { #>
                                            <span class="skill_pric-price">{{{ panel.discounted_price }}}</span>
                                        <# } #>
                                    </div>
                                </div>
                                <div class="skill_pric-list-daata">
                                    <# if (panel.features_list) { #>
                                        <ul>
                                            <# var features = panel.features_list.split('\n'); #>
                                            <# _.each(features, function(feature) { #>
                                                <# if (feature.trim()) { #>
                                                    <li><i class="fas fa-link"></i>{{{ feature.trim() }}}</li>
                                                <# } #>
                                            <# }); #>
                                        </ul>
                                    <# } #>
                                </div>
                                <div class="skill_pric-button">
                                    <# if (panel.button_text) { #>
                                        <a href="{{{ panel.button_link.url || '#' }}}" class="skill_pric-btn">
                                            {{{ panel.button_text }}}
                                        </a>
                                    <# } #>
                                </div>
                            </div>
                            <div class="bar"></div>
                        </div>
                    </div>
                <# }); #>
            </div>
        <# } #>
        <?php
    }
}