<?php

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

class skill_project_timeline_Widget extends \Elementor\Widget_Base {
    public function get_name() {
        return 'skill_project_timeline_Widget';
    }
    public function get_title() {
        return esc_html__('زمانبندی پروژه', 'skill');
    }
    public function get_icon() {
        return 'eicon-time-line';
    }
    public function get_categories() {
        return ['skill-category'];
    }
    
    protected function register_controls() {
        $this->start_controls_section(
            'content_section',
            [
                'label' => esc_html__('تنظیمات زمانبندی', 'skill'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'section_title',
            [
                'label' => esc_html__('عنوان بخش', 'skill'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => esc_html__('مراحل پروژه', 'skill'),
                'placeholder' => esc_html__('عنوان بخش را وارد کنید', 'skill'),
            ]
        );

        $this->add_control(
            'section_description',
            [
                'label' => esc_html__('توضیحات', 'skill'),
                'type' => \Elementor\Controls_Manager::TEXTAREA,
                'default' => esc_html__('مراحل مختلف انجام پروژه از شروع تا پایان', 'skill'),
                'placeholder' => esc_html__('توضیحات زمانبندی را وارد کنید', 'skill'),
            ]
        );

        $this->add_control(
            'timeline_style',
            [
                'label' => esc_html__('سبک نمایش', 'skill'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'vertical',
                'options' => [
                    'vertical' => esc_html__('عمودی', 'skill'),
                    'horizontal' => esc_html__('افقی', 'skill'),
                ],
            ]
        );

        $this->end_controls_section();

        // Style Tab
        $this->start_controls_section(
            'style_section',
            [
                'label' => esc_html__('استایل', 'skill'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'title_color',
            [
                'label' => esc_html__('رنگ عنوان', 'skill'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .timeline-title' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'title_typography',
                'selector' => '{{WRAPPER}} .timeline-title',
            ]
        );

        $this->add_control(
            'date_color',
            [
                'label' => esc_html__('رنگ تاریخ', 'skill'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .timeline-date' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'event_title_color',
            [
                'label' => esc_html__('رنگ عنوان رویداد', 'skill'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .timeline-event-title' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'event_description_color',
            [
                'label' => esc_html__('رنگ توضیحات رویداد', 'skill'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .timeline-event-description' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'timeline_line_color',
            [
                'label' => esc_html__('رنگ خط زمانبندی', 'skill'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .timeline-line' => 'background-color: {{VALUE}};',
                    '{{WRAPPER}} .timeline-dot' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        
        $post_id = get_the_ID();
        $timeline_events = get_post_meta($post_id, '_project_timeline', true);
        
        if (!is_array($timeline_events) || empty($timeline_events)) {
            return;
        }

        // Helper function to convert Gregorian to Jalali
        function convert_to_jalali($gregorian_date) {
            if (empty($gregorian_date)) {
                return '';
            }
            
            $date_parts = explode('-', $gregorian_date);
            if (count($date_parts) === 3) {
                $year = (int)$date_parts[0];
                $month = (int)$date_parts[1];
                $day = (int)$date_parts[2];
                
                // Persian month names
                $persian_months = [
                    1 => 'فروردین', 2 => 'اردیبهشت', 3 => 'خرداد',
                    4 => 'تیر', 5 => 'مرداد', 6 => 'شهریور',
                    7 => 'مهر', 8 => 'آبان', 9 => 'آذر',
                    10 => 'دی', 11 => 'بهمن', 12 => 'اسفند'
                ];
                
                // Accurate Persian calendar conversion
                $gy = $year;
                $gm = $month;
                $gd = $day;
                
                $g_d_m = [0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334];
                $gy2 = ($gm > 2) ? ($gy + 1) : $gy;
                $days = 355666 + (365 * $gy) + ((int)(($gy2 + 3) / 4)) - ((int)(($gy2 + 99) / 100)) + ((int)(($gy2 + 399) / 400)) + $gd + $g_d_m[$gm - 1];
                $jy = -1595 + (33 * ((int)($days / 12053)));
                $days %= 12053;
                $jy += 4 * ((int)($days / 1461));
                $days %= 1461;
                if ($days > 365) {
                    $jy += (int)(($days - 1) / 365);
                    $days = ($days - 1) % 365;
                }
                if ($days < 186) {
                    $jm = 1 + (int)($days / 31);
                    $jd = 1 + ($days % 31);
                } else {
                    $jm = 7 + (int)(($days - 186) / 30);
                    $jd = 1 + (($days - 186) % 30);
                }
                
                // Get Persian month name
                $persian_month_name = $persian_months[$jm] ?? $jm;
                
                // Return in Persian format
                return $jd . ' ' . $persian_month_name . ' ' . $jy;
            }
            
            return $gregorian_date;
        }

        $timeline_style = $settings['timeline_style'];
        ?>
        <div class="skill-project-timeline-widget">
            <div class="timeline-container timeline-<?php echo esc_attr($timeline_style); ?>">
                <?php if ($settings['section_title']): ?>
                    <h3 class="timeline-title"><?php echo esc_html($settings['section_title']); ?></h3>
                <?php endif; ?>
                
                <?php if ($settings['section_description']): ?>
                    <p class="timeline-description"><?php echo esc_html($settings['section_description']); ?></p>
                <?php endif; ?>
                
                <div class="timeline-events">
                    <?php foreach ($timeline_events as $index => $event): 
                        $side = ($index % 2 === 0) ? 'left' : 'right';
                    ?>
                        <div class="timeline-row">
                            <?php if ($side === 'left'): ?>
                                <div class="timeline-card side-left">
                                    <div class="card-head">
                                        <div class="window-dots"><span></span><span></span><span></span></div>
                                    </div>
                                    <div class="card-body">
                                        <h4 class="card-title"><?php echo esc_html($event['title']); ?></h4>
                                        <?php if (!empty($event['date'])): ?>
                                            <div class="card-date"><?php echo esc_html(convert_to_jalali($event['date'])); ?></div>
                                        <?php endif; ?>
                                        <?php if (!empty($event['description'])): ?>
                                            <p class="card-text"><?php echo esc_html($event['description']); ?></p>
                                        <?php endif; ?>
                                    </div>
                                </div>
                                <div class="timeline-dot"></div>
                                <div class="timeline-spacer"></div>
                            <?php else: ?>
                                <div class="timeline-spacer"></div>
                                <div class="timeline-dot"></div>
                                <div class="timeline-card side-right">
                                    <div class="card-head">
                                        <div class="window-dots"><span></span><span></span><span></span></div>
                                    </div>
                                    <div class="card-body">
                                        <h4 class="card-title"><?php echo esc_html($event['title']); ?></h4>
                                        <?php if (!empty($event['date'])): ?>
                                            <div class="card-date"><?php echo esc_html(convert_to_jalali($event['date'])); ?></div>
                                        <?php endif; ?>
                                        <?php if (!empty($event['description'])): ?>
                                            <p class="card-text"><?php echo esc_html($event['description']); ?></p>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            <?php endif; ?>
                        </div>
                    <?php endforeach; ?>
                </div>
                
                <div class="timeline-line"></div>
            </div>
        </div>

        <style>
        .skill-project-timeline-widget {
            margin: 4rem 0;
        }
        
        .timeline-container {
            position: relative;
            padding: 3rem 0;
            max-width: 100%;
            background: linear-gradient(135deg, #e0f2fe 0%, #f0f9ff 100%);
            border-radius: 20px;
            margin: 2rem 0;
        }
        
        .timeline-title {
            margin-bottom: 3rem;
            font-size: 2.5rem;
            font-weight: 700;
            color: #1e293b;
            text-align: center;
            position: relative;
            letter-spacing: -0.02em;
        }
        
        .timeline-description {
            margin-bottom: 4rem;
            color: #64748b;
            text-align: center;
            font-size: 1.1rem;
            line-height: 1.6;
            max-width: 600px;
            margin-left: auto;
            margin-right: auto;
            font-weight: 400;
        }
        
        /* Vertical centered line with alternating cards (exactly like the image) */
        .timeline-events {
            position: relative;
            padding: 1rem 0 2rem;
        }
        .timeline-events::before {
            content: '';
            position: absolute;
            top: 0;
            bottom: 0;
            left: 50%;
            transform: translateX(-50%);
            width: 4px;
            border-radius: 2px;
            background: linear-gradient(
                to bottom,
                rgba(255,255,255,0.95) 0%,
                rgba(255,255,255,0.95) 60%,
                rgba(255,255,255,0.4) 60%,
                rgba(255,255,255,0.4) 100%
            );
        }
        .timeline-row {
            position: relative;
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 6rem;
            align-items: center;
            margin: 2.5rem 0;
        }
        .timeline-dot {
            position: absolute;
            left: 50%;
            top: 50%;
            transform: translate(-50%, -50%);
            width: 14px;
            height: 14px;
            background: #22d3ee;
            border: 4px solid #ffffff;
            border-radius: 999px;
            box-shadow: 0 0 0 4px rgba(255,255,255,0.25), 0 6px 14px rgba(0,0,0,0.15);
            z-index: 2;
        }
        .timeline-card {
            background: #ffffff;
            border-radius: 14px;
            box-shadow: 0 10px 25px rgba(2, 6, 23, 0.12);
            width: 100%;
            max-width: 420px;
            overflow: hidden;
            border: 1px solid rgba(255,255,255,0.6);
        }
        .timeline-card.side-left { justify-self: end; }
        .timeline-card.side-right { justify-self: start; }
        .card-head {
            height: 38px;
            background: linear-gradient(135deg, #8b5cf6 0%, #3b82f6 100%);
            display: flex;
            align-items: center;
            padding: 0 12px;
        }
        .window-dots { display: flex; gap: 6px; }
        .window-dots span {
            width: 10px; height: 10px; border-radius: 999px; background: rgba(255,255,255,0.9);
        }
        .card-body { padding: 16px 18px 18px; }
        .card-title { font-size: 1.05rem; font-weight: 700; color: #1f2937; margin: 0 0 6px; }
        .card-date { font-size: 0.85rem; color: #64748b; margin-bottom: 10px; }
        .card-text { font-size: 0.95rem; color: #475569; line-height: 1.65; margin: 0; }
        
        /* Responsive */
        @media (max-width: 992px) {
            .timeline-row { gap: 3rem; }
        }
        @media (max-width: 768px) {
            .timeline-row { grid-template-columns: 1fr; }
            .timeline-card.side-left,
            .timeline-card.side-right { justify-self: stretch; }
        }
        @media (max-width: 480px) {
            .card-head { height: 34px; }
            .card-title { font-size: 1rem; }
        }
        </style>
        <?php
    }
}
