<?php
/**
 * Skill Text Gradient Widget
 * 
 * A custom Elementor widget that displays text with gradient colors
 * 
 * @package Skill
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

class Skill_Text_Gradient_Widget extends \Elementor\Widget_Base {

    /**
     * Get widget name
     */
    public function get_name() {
        return 'skill_text_gradient';
    }

    /**
     * Get widget title
     */
    public function get_title() {
        return esc_html__('متن گرادینت', 'skill');
    }

    /**
     * Get widget icon
     */
    public function get_icon() {
        return 'eicon-text';
    }

    /**
     * Get widget categories
     */
    public function get_categories() {
        return ['skill-category'];
    }

    /**
     * Get widget keywords
     */
    public function get_keywords() {
        return ['text', 'gradient', 'color', 'متن', 'گرادینت', 'رنگ'];
    }

    /**
     * Register widget controls
     */
    protected function register_controls() {

        // Content Section
        $this->start_controls_section(
            'content_section',
            [
                'label' => esc_html__('محتوای متن', 'skill'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'text_content',
            [
                'label' => esc_html__('متن', 'skill'),
                'type' => \Elementor\Controls_Manager::TEXTAREA,
                'default' => esc_html__('متن گرادینت شما اینجا', 'skill'),
                'placeholder' => esc_html__('متن خود را وارد کنید', 'skill'),
                'rows' => 3,
            ]
        );

        $this->add_control(
            'html_tag',
            [
                'label' => esc_html__('تگ HTML', 'skill'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'h2',
                'options' => [
                    'h1' => 'H1',
                    'h2' => 'H2',
                    'h3' => 'H3',
                    'h4' => 'H4',
                    'h5' => 'H5',
                    'h6' => 'H6',
                    'p' => 'P',
                    'span' => 'SPAN',
                    'div' => 'DIV',
                ],
            ]
        );

        $this->end_controls_section();

        // Gradient Colors Section
        $this->start_controls_section(
            'gradient_section',
            [
                'label' => esc_html__('رنگ‌های گرادینت', 'skill'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'gradient_color_1',
            [
                'label' => esc_html__('رنگ اول', 'skill'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#984300',
            ]
        );

        $this->add_control(
            'gradient_color_2',
            [
                'label' => esc_html__('رنگ دوم', 'skill'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#FD6F00',
            ]
        );

        $this->add_control(
            'gradient_color_3',
            [
                'label' => esc_html__('رنگ سوم', 'skill'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#CA5900',
            ]
        );

        $this->add_control(
            'gradient_direction',
            [
                'label' => esc_html__('جهت گرادینت', 'skill'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'to bottom',
                'options' => [
                    'to bottom' => esc_html__('به پایین', 'skill'),
                    'to top' => esc_html__('به بالا', 'skill'),
                    'to right' => esc_html__('به راست', 'skill'),
                    'to left' => esc_html__('به چپ', 'skill'),
                    'to bottom right' => esc_html__('به پایین راست', 'skill'),
                    'to bottom left' => esc_html__('به پایین چپ', 'skill'),
                    'to top right' => esc_html__('به بالا راست', 'skill'),
                    'to top left' => esc_html__('به بالا چپ', 'skill'),
                ],
            ]
        );

        $this->end_controls_section();

        // Typography Section
        $this->start_controls_section(
            'typography_section',
            [
                'label' => esc_html__('تایپوگرافی', 'skill'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'typography',
                'selector' => '{{WRAPPER}} .skill-gradient-text',
            ]
        );

        $this->add_responsive_control(
            'text_align',
            [
                'label' => esc_html__('تراز متن', 'skill'),
                'type' => \Elementor\Controls_Manager::CHOOSE,
                'options' => [
                    'left' => [
                        'title' => esc_html__('چپ', 'skill'),
                        'icon' => 'eicon-text-align-left',
                    ],
                    'center' => [
                        'title' => esc_html__('وسط', 'skill'),
                        'icon' => 'eicon-text-align-center',
                    ],
                    'right' => [
                        'title' => esc_html__('راست', 'skill'),
                        'icon' => 'eicon-text-align-right',
                    ],
                    'justify' => [
                        'title' => esc_html__('تمام عرض', 'skill'),
                        'icon' => 'eicon-text-align-justify',
                    ],
                ],
                'default' => 'center',
                'selectors' => [
                    '{{WRAPPER}} .skill-gradient-text' => 'text-align: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Spacing Section
        $this->start_controls_section(
            'spacing_section',
            [
                'label' => esc_html__('فاصله‌گذاری', 'skill'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'margin',
            [
                'label' => esc_html__('حاشیه', 'skill'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .skill-gradient-text' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'padding',
            [
                'label' => esc_html__('پدینگ', 'skill'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .skill-gradient-text' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();
    }

    /**
     * Render widget output on the frontend
     */
    protected function render() {
        $settings = $this->get_settings_for_display();

        // Get gradient colors
        $color1 = $settings['gradient_color_1'] ?: '#984300';
        $color2 = $settings['gradient_color_2'] ?: '#FD6F00';
        $color3 = $settings['gradient_color_3'] ?: '#CA5900';
        $direction = $settings['gradient_direction'] ?: 'to bottom';

        // Get HTML tag
        $tag = $settings['html_tag'] ?: 'h2';

        // Generate unique ID for this widget instance
        $unique_id = 'skill-gradient-' . $this->get_id();

        // Output the text with inline styles
        echo '<' . esc_attr($tag) . ' class="skill-gradient-text" id="' . esc_attr($unique_id) . '">';
        echo wp_kses_post($settings['text_content']);
        echo '</' . esc_attr($tag) . '>';

        // Output CSS for gradient
        echo '<style>
            #' . esc_attr($unique_id) . ' {
                background: -webkit-linear-gradient(' . esc_attr($direction) . ', ' . esc_attr($color1) . ', ' . esc_attr($color2) . ', ' . esc_attr($color3) . ');
                background: linear-gradient(' . esc_attr($direction) . ', ' . esc_attr($color1) . ', ' . esc_attr($color2) . ', ' . esc_attr($color3) . ');
                -webkit-background-clip: text;
                background-clip: text;
                -webkit-text-fill-color: transparent;
                color: transparent;
                display: inline-block;
            }
        </style>';
    }

    /**
     * Render widget output in the editor
     */
    protected function content_template() {
        ?>
        <#
        var color1 = settings.gradient_color_1 || '#984300';
        var color2 = settings.gradient_color_2 || '#FD6F00';
        var color3 = settings.gradient_color_3 || '#CA5900';
        var direction = settings.gradient_direction || 'to bottom';
        var tag = settings.html_tag || 'h2';
        var uniqueId = 'skill-gradient-' + view.getID();
        #>
        
        <{{ tag }} class="skill-gradient-text" id="{{ uniqueId }}">
            {{{ settings.text_content }}}
        </{{ tag }}>
        
        <style>
            #{{ uniqueId }} {
                background: -webkit-linear-gradient({{ direction }}, {{ color1 }}, {{ color2 }}, {{ color3 }});
                background: linear-gradient({{ direction }}, {{ color1 }}, {{ color2 }}, {{ color3 }});
                -webkit-background-clip: text;
                background-clip: text;
                -webkit-text-fill-color: transparent;
                color: transparent;
                display: inline-block;
            }
        </style>
        <?php
    }
}
